<?php
namespace boru\dweb\Routing;

use boru\dweb\Http\Request;
use boru\dweb\Config\ConfigKeys;
use boru\dweb\Contracts\SettingsInterface;

class WebRouter
{
    /** @var SettingsInterface */
    private $config;

    /** @var RouteCollection */
    private $routes;

    public function __construct(SettingsInterface $config, RouteCollection $routes)
    {
        $this->config = $config;
        $this->routes = $routes;
    }

    public function dispatch(Request $req)
    {
        // ACTIONS
        $actionParam = $this->config->get(ConfigKeys::ACTION_PARAM, 'action');
        $action = $req->param($actionParam, null);
        if ($action) {
            $action = $this->qualifyIfNeeded($action, $this->config->get(ConfigKeys::DEFAULT_MODULE, null));

            $record = $this->routes->getAction($action);
            if ($record && $this->actionAllowsMethod($record, $req->method())) {
                return call_user_func($record['handler'], $req);
            }

            throw new \boru\dweb\Http\HttpException(
                $record ? 405 : 404,
                $record ? 'Method Not Allowed' : 'Not Found',
                ($record ? 'Method not allowed for action: ' : 'Action not found: ') . $action
            );
        }

        // VIEWS
        $pageParam = $this->config->get(ConfigKeys::PAGE_PARAM, 'view');
        $defaultView = $this->config->get(ConfigKeys::DEFAULT_VIEW, '');
        $view = $req->param($pageParam, $defaultView);
        $view = $this->qualifyIfNeeded($view, $this->config->get(ConfigKeys::DEFAULT_MODULE, null));
        $handler = $this->routes->getView($view);

        if ($handler) {
            return call_user_func($handler, $req);
        }

        // fallback to "" (root) if a module registered it as default
        $fallback = $this->routes->getView('');
        if ($fallback) {
            return call_user_func($fallback, $req);
        }
        throw new \boru\dweb\Http\NotFoundException('View not found: ' . $view);
    }

    private function qualifyIfNeeded($id, $defaultModule)
    {
        $id = (string)$id;
        if ($id === '') return $id;

        // already qualified
        if (strpos($id, '.') !== false) return $id;

        // if no module and no default, leave as-is (will 404 deterministically)
        if (!$defaultModule) return $id;

        return $defaultModule . '.' . $id;
    }

    private function actionAllowsMethod($actionRecord, $method)
    {
        if (!$actionRecord) return false;
        if (!isset($actionRecord['methods']) || $actionRecord['methods'] === null) return true;
        return isset($actionRecord['methods'][strtoupper((string)$method)]);
    }


}
