# Error Handling in boru/dweb

Error handling in `boru/dweb` is **centralized, explicit, and content-aware**.
All errors flow through a single responder and are rendered consistently
across HTML, JSON, and API contexts.

This document explains how errors are created, propagated, and rendered.

---

## Core Principles

- Errors are handled centrally
- Controllers do not render errors
- Content type matters
- Debug output is configurable
- Sanitization is intentional

---

## The ExceptionResponder

All uncaught exceptions are converted into responses by:

```

boru\dweb\Http\ExceptionResponder

```

The responder:
- determines HTTP status
- decides output format (HTML vs JSON)
- applies debug/sanitization rules
- unwraps nested exceptions
- produces a Response object

---

## Exception Flow

```

Request
→ Middleware
→ Router
→ View / Action
→ throws Exception
←
←
→ ExceptionResponder
→ Response

```

Controllers do not catch framework-level errors.

---

## HttpException

Framework-level HTTP errors should use:

```

HttpException

````

Features:
- HTTP status code
- user-facing message
- optional debug details
- optional wrapped exception

Example:

```php
throw new HttpException(
    404,
    'Not Found',
    'The requested item does not exist.'
);
````

---

## Wrapping Exceptions

`HttpException` may wrap a lower-level exception:

```php
throw new HttpException(
    500,
    'Service Error',
    'Unable to process request.',
    $e
);
```

The responder will:

* show the user message
* use the wrapped exception for debug output
* correctly report the HTTP status

---

## Status Code Resolution

The responder determines status codes in this order:

1. Explicit HttpException status
2. `getStatusCode()` if present
3. Known exception types (e.g. InvalidArgumentException → 400)
4. Heuristics (initialization / publish errors → 503)
5. Default → 500

---

## Content-Type Awareness

### Action / API Requests

Detected by:

* `action` parameter
* `Accept: application/json`
* `Accept: text/event-stream`
* `Content-Type: application/json`

Response:

* JSON payload
* structured error fields
* optional debug details

Example JSON:

```json
{
  "error": true,
  "message": "Invalid JSON body.",
  "code": 0,
  "exception": "InvalidArgumentException",
  "details": "Syntax error"
}
```

---

### View / HTML Requests

Response:

* HTML error page
* sectioned output
* optional stack trace

Sections may include:

* Message
* Exception
* Location
* Wrapped As
* Details
* Trace

---

## Debug Configuration

Error behavior is controlled by settings:

| Setting                              | Purpose                       |
| ------------------------------------ | ----------------------------- |
| `dweb.debug`                         | Master debug switch           |
| `dweb.errors.show_trace`             | Include stack traces          |
| `dweb.errors.sanitize_user_messages` | Sanitize user-facing messages |
| `dweb.errors.sanitize_debug_details` | Sanitize debug output         |

These flags apply consistently across HTML and JSON responses.

---

## Sanitization

Sanitization protects against:

* leaking sensitive data
* unsafe HTML output
* internal file paths in production

Sanitization is applied to:

* user messages
* debug details
* stack traces (optional)

---

## Exception Origin Tracking

When an `HttpException` wraps another exception:

* The *origin* exception is used for:

  * class name
  * location
  * trace
* The wrapper is still reported as:

  * HTTP status
  * error category

This avoids losing the real cause.

---

## Logging

Unhandled exceptions are logged via:

```
LoggerInterface
```

Logging happens:

* once per exception
* before response rendering
* without interfering with output

Default behavior:

* error_log in debug mode
* NullLogger in production unless overridden

---

## Middleware and Errors

Middleware may:

* throw HttpException
* throw generic exceptions

All exceptions propagate upward.

Middleware should not:

* render error responses
* catch exceptions unless transforming them

---

## Controller Responsibilities

Controllers should:

* throw HttpException for client errors
* let unexpected errors bubble

Controllers should not:

* catch framework exceptions
* format error responses
* emit headers

---

## Debug Route Safety

Debug endpoints:

* exist only when `dweb.debug = true`
* respect error handling rules
* do not bypass ExceptionResponder

---

## Design Rules (Important)

* One error pipeline
* One responder
* No per-module error handlers
* No magic fallbacks
* No silent failures

If an error occurs, it should be:

* visible
* inspectable
* debuggable
* consistent

---

## Summary

Error handling in `boru/dweb` is:

* centralized
* predictable
* secure
* content-aware
* configurable

Errors are first-class framework behavior,
not something modules need to manage.
