# Testing in boru/dweb

Testing in `boru/dweb` is **explicit, layered, and pragmatic**.
The framework favors **smoke-style integration tests** over heavy mocking,
with a small number of deterministic entry points.

This document explains how testing works and how it is intended to evolve.

---

## Core Philosophy

- Prefer real execution over mocks
- Test the framework as it is actually used
- Catch wiring and lifecycle bugs early
- Avoid test frameworks unless needed
- Keep tests readable and debuggable

If something breaks, the test output should tell you *what* and *why*.

---

## Test Layers

dweb testing is intentionally layered:

```

Unit (optional)
→ Router-level smoke tests
→ WebUI end-to-end tests

```

Each layer catches different classes of issues.

---

## Smoke Tests (Primary)

Smoke tests are the **primary validation mechanism** for dweb.

They:
- exercise real routing
- use real middleware
- use real modules
- use real renderers
- avoid mocking internals

Smoke tests are run via CLI and require no web server.

---

## Smoke Test Structure

Typical layout:

```

tests/
smoke/
_lib.php
_bootstrap.php
01_default_route.php
02_path_view.php
03_query_view.php
04_action_get.php
05_method_override.php
06_json_body.php
07_security_headers.php
08_route_dump.php
09_webui_render.php
run.php

````

Each file:
- tests one behavior
- fails fast
- prints a clear reason

---

## CLI Smoke Runner

Smoke tests are executed with:

```bash
php tests/smoke/run.php
````

The runner:

* executes tests in numeric order
* stops on first failure
* prints a final success message

This makes it ideal for:

* CI pipelines
* quick local validation
* refactor safety checks

---

## Router-Level Smoke Tests

Router-level tests:

* build the router pipeline directly
* bypass WebUI
* validate routing, middleware, and responses

These tests catch:

* route registration bugs
* method constraints
* middleware ordering issues
* debug endpoint availability

They are fast and deterministic.

---

## WebUI End-to-End Tests

WebUI smoke tests:

* execute `WebUI->render()`
* simulate real HTTP requests
* capture output via buffering

These tests catch:

* container lifecycle issues
* middleware wiring errors
* debug endpoint visibility
* response emission behavior

They are intentionally few but high-value.

---

## Request Simulation

Requests are simulated by:

* constructing `Request` objects directly (router tests)
* populating `$_SERVER`, `$_GET`, `$_POST` (WebUI tests)

This avoids:

* HTTP clients
* test servers
* brittle network dependencies

---

## Assertions

Assertions are minimal and explicit.

Typical assertions include:

* HTTP status code
* presence of known strings
* JSON decode success
* header existence

Tests avoid:

* fragile string matching
* snapshot testing
* exact HTML comparisons

The goal is confidence, not pixel perfection.

---

## Error Case Testing

Error paths are tested implicitly by:

* method constraint failures (405)
* missing routes (404)
* invalid JSON bodies (400)

These validate:

* ExceptionResponder behavior
* content-type detection
* debug flags

---

## Modules in Tests

The `Skeleton` module serves as:

* a reference implementation
* a validation harness
* documentation by example

Smoke tests depend on Skeleton intentionally.
If Skeleton breaks, the framework contract has changed.

---

## Unit Tests (Optional)

Unit tests are optional and not required.

They may be useful for:

* low-level utilities
* pure functions
* complex algorithms

They are not required for:

* routing
* rendering
* middleware
* error handling

---

## Why Not PHPUnit (By Default)

The framework intentionally avoids requiring:

* PHPUnit
* mocks
* test doubles
* annotations

This keeps:

* contributor barrier low
* test execution simple
* framework behavior visible

Nothing prevents adding PHPUnit later if needed.

---

## CI Integration

Recommended CI steps:

```bash
composer install
php tests/smoke/run.php
```

If smoke tests pass:

* the framework is wired correctly
* major regressions are unlikely

---

## Design Rules (Important)

* Tests should reflect real usage
* Smoke tests must be deterministic
* Debug output should aid diagnosis
* No silent failures
* No hidden setup

If a test is confusing, it should be simplified.

---

## Summary

Testing in `boru/dweb` is:

* integration-first
* explicit
* fast
* readable
* reliable

Smoke tests provide confidence that the framework works as a whole,
not just in isolated pieces.