<?php
namespace boru\dweb\Http;

/**
 * Small helper for HTMX response headers.
 * Explicit + opt-in. Safe for non-HTMX callers too.
 *
 * Docs: https://htmx.org/reference/#response_headers
 */
class HxResponse
{
    /**
     * HX-Redirect: full redirect (client-side)
     * @param HtmlResponse $res
     * @param string $url
     * @return HtmlResponse
     */
    public static function redirect(HtmlResponse $res, $url)
    {
        $res->setHeader('HX-Redirect', (string)$url);
        return $res;
    }

    /**
     * HX-Refresh: "true" to refresh the page
     * @param HtmlResponse $res
     * @param bool $value
     * @return HtmlResponse
     */
    public static function refresh(HtmlResponse $res, $value = true)
    {
        $res->setHeader('HX-Refresh', $value ? 'true' : 'false');
        return $res;
    }

    /**
     * HX-Location: client-side navigation (may include target/swap/etc)
     * Accepts string URL OR array payload (JSON encoded).
     *
     * @param HtmlResponse $res
     * @param mixed $location string|array
     * @return HtmlResponse
     */
    public static function location(HtmlResponse $res, $location)
    {
        if (is_array($location)) {
            $res->setHeader('HX-Location', self::json($location));
        } else {
            $res->setHeader('HX-Location', (string)$location);
        }
        return $res;
    }

    /**
     * HX-Push-Url: push URL into history (true|string|false)
     * @param HtmlResponse $res
     * @param mixed $value
     * @return HtmlResponse
     */
    public static function pushUrl(HtmlResponse $res, $value)
    {
        if ($value === true) $value = 'true';
        if ($value === false) $value = 'false';
        $res->setHeader('HX-Push-Url', (string)$value);
        return $res;
    }

    /**
     * HX-Replace-Url: replace URL in history (true|string|false)
     * @param HtmlResponse $res
     * @param mixed $value
     * @return HtmlResponse
     */
    public static function replaceUrl(HtmlResponse $res, $value)
    {
        if ($value === true) $value = 'true';
        if ($value === false) $value = 'false';
        $res->setHeader('HX-Replace-Url', (string)$value);
        return $res;
    }

    /**
     * HX-Reswap: change swap strategy, e.g. "outerHTML"
     * @param HtmlResponse $res
     * @param string $strategy
     * @return HtmlResponse
     */
    public static function reswap(HtmlResponse $res, $strategy)
    {
        $res->setHeader('HX-Reswap', (string)$strategy);
        return $res;
    }

    /**
     * HX-Retarget: change target selector, e.g. "#content"
     * @param HtmlResponse $res
     * @param string $selector
     * @return HtmlResponse
     */
    public static function retarget(HtmlResponse $res, $selector)
    {
        $res->setHeader('HX-Retarget', (string)$selector);
        return $res;
    }

    /**
     * HX-Trigger / HX-Trigger-After-Settle / HX-Trigger-After-Swap
     * $event can be:
     *  - string event name
     *  - array of eventName => payload
     *
     * @param HtmlResponse $res
     * @param mixed $event
     * @param string $when "now"|"afterSwap"|"afterSettle"
     * @return HtmlResponse
     */
    public static function trigger(HtmlResponse $res, $event, $when = 'now')
    {
        $header = 'HX-Trigger';
        if ($when === 'afterSwap') $header = 'HX-Trigger-After-Swap';
        if ($when === 'afterSettle') $header = 'HX-Trigger-After-Settle';

        if (is_array($event)) {
            $res->setHeader($header, self::json($event));
        } else {
            $res->setHeader($header, (string)$event);
        }
        return $res;
    }

    /**
     * Convenience: tag a response as a fragment explicitly (header already added by AbstractView::fragment()).
     * This is useful for Actions that return HTML fragments too.
     *
     * @param HtmlResponse $res
     * @param string $qualified e.g. "Skeleton.hello"
     * @return HtmlResponse
     */
    public static function fragment(HtmlResponse $res, $qualified)
    {
        $res->setHeader('X-DWeb-Fragment', (string)$qualified);
        return $res;
    }

    /**
     * Explicit helper to target a section selector + swap strategy for HTMX.
     * @param HtmlResponse $res
     * @param string $selector e.g. "#section-sidebar"
     * @param string $swap e.g. "outerHTML"
     * @return HtmlResponse
     */
    public static function target(HtmlResponse $res, $selector, $swap = 'outerHTML')
    {
        self::retarget($res, (string)$selector);
        self::reswap($res, (string)$swap);
        return $res;
    }

    private static function json(array $data)
    {
        // PHP 5.6 safe JSON encoding (no JSON_THROW_ON_ERROR)
        $json = json_encode($data);
        return $json !== false ? $json : '{}';
    }
}
