<?php
namespace boru\dweb\Mvc;

use boru\dweb\Contracts\RendererInterface;
use boru\dweb\Contracts\ModuleInterface;
use boru\dweb\Routing\CanonicalUrl;
use boru\dweb\Mvc\DwebHelper;

abstract class AbstractControllerFactory
{
    /** @var RendererInterface */
    protected $renderer;

    /** @var CanonicalUrl */
    protected $canonical;

    /** @var ModuleInterface */
    protected $module;

    /** @var ModuleNamespaceResolver */
    protected $resolver;

    public function __construct(
        RendererInterface $renderer,
        CanonicalUrl $canonical,
        ModuleInterface $module,
        ModuleNamespaceResolver $resolver
    ) {
        $this->renderer  = $renderer;
        $this->canonical = $canonical;
        $this->module    = $module;
        $this->resolver  = $resolver;
    }

    /**
     * @return string e.g. "Skeleton"
     */
    protected function moduleName()
    {
        return (string)$this->module->getName();
    }

    /**
     * @param string $shortName like "home" or "ping"
     * @return object
     */
    protected function makeController($shortName)
    {
        $moduleName = $this->moduleName();

        $suffix = $this->subNamespace() . '\\' . $this->studly($shortName) . $this->classSuffix();
        $class  = $this->resolver->qualify($this->module, $suffix);

        if (!class_exists($class)) {
            throw new \RuntimeException('Controller class not found: ' . $class);
        }

        // Current dweb convention: controllers accept (RendererInterface, moduleName)
        $controller = new $class($this->renderer, $moduleName);

        // Optional injection point for helpers
        if (method_exists($controller, 'setDwebHelper')) {
            $controller->setDwebHelper(new DwebHelper($this->canonical, $moduleName));
        }

        return $controller;
    }

    abstract protected function subNamespace(); // "Views" | "Actions"
    abstract protected function classSuffix();  // "View" | "Action"

    protected function studly($name)
    {
        $name = preg_replace('/[^a-zA-Z0-9_]/', '_', (string)$name);
        $parts = preg_split('/_+/', $name);
        $out = '';
        foreach ($parts as $p) {
            if ($p === '') continue;
            $out .= strtoupper(substr($p, 0, 1)) . substr($p, 1);
        }
        return $out;
    }
}
