<?php
namespace boru\dweb\Providers\Auth;

use boru\dweb\Contracts\AuthProviderInterface;
use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Domain\UserIdentity;

/**
 * VtigerSessionAuthenticator
 *
 * Builds a UserIdentity from Vtiger session state.
 * This is UI/auth plumbing, not BoruAI.
 */
class VtigerSessionAuthenticator implements AuthProviderInterface
{
    /** @var SettingsInterface|null */
    private $settings;

    /** @var string */
    private $prefix;

    /**
     * @param SettingsInterface|null $settings
     * @param string $settingsPrefix
     */
    public function __construct(SettingsInterface $settings = null, $settingsPrefix = 'dweb.auth.vtiger.')
    {
        $this->settings = $settings;
        $this->prefix = (string)$settingsPrefix;
    }

    /**
     * @return UserIdentity|null
     */
    public function currentUser()
    {
        // If sessions aren't started, attempt to start (safe if already started)
        if (function_exists('session_status')) {
            if (session_status() === PHP_SESSION_NONE) {
                @session_start();
            }
        } else {
            // PHP 5.6 fallback: if no session id, attempt start
            if (session_id() === '') {
                @session_start();
            }
        }

        // Configurable session keys (defaults match common vtiger patterns)
        $idKey   = (string)$this->get('session_user_id_key', 'authenticated_user_id');
        $nameKey = (string)$this->get('session_user_name_key', 'authenticated_user_name');

        $userId = isset($_SESSION[$idKey]) ? $_SESSION[$idKey] : null;
        $userName = isset($_SESSION[$nameKey]) ? $_SESSION[$nameKey] : null;

        // Some vtiger installs only have user id; name can be derived or left generic
        if ($userId !== null && $userId !== '') {
            if ($userName === null || $userName === '') {
                $userName = (string)$this->get('default_user_name', 'User');
            }
            return new UserIdentity($userId, (string)$userName);
        }

        // Optional: support alternate vtiger keys
        $altIdKey = (string)$this->get('session_user_id_key_alt', '');
        if ($altIdKey !== '' && isset($_SESSION[$altIdKey]) && $_SESSION[$altIdKey] !== '') {
            $userId = $_SESSION[$altIdKey];
            if ($userName === null || $userName === '') {
                $userName = (string)$this->get('default_user_name', 'User');
            }
            return new UserIdentity($userId, (string)$userName);
        }

        // Optional public fallback (handy for dev / internal deployments)
        if ((bool)$this->get('allow_public_fallback', false)) {
            $id = $this->get('public_user_id', 1);
            $name = (string)$this->get('public_user_name', 'Public');
            return new UserIdentity($id, $name);
        }

        return null;
    }

    private function get($key, $default = null)
    {
        if (!$this->settings) return $default;
        return $this->settings->get($this->prefix . $key, $default);
    }
}
