<?php
namespace boru\dweb\Routing;

use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Http\Request;
use boru\dweb\Config\ConfigKeys;

class CanonicalUrl
{
    /** @var SettingsInterface */
    private $settings;

    /** @var Request */
    private $request;

    public function __construct(SettingsInterface $settings, Request $request)
    {
        $this->settings = $settings;
        $this->request  = $request;
    }

    public function enabled()
    {
        return (bool)$this->settings->get('dweb.urls.canonical.enabled', false);
    }

    /**
     * @param string $qualified "Module.view"
     * @param array|string|null $tail
     * @return string
     */
    public function view($qualified, $tail = null)
    {
        $qualified = (string)$qualified;

        if ($this->enabled()) {
            list($module, $name) = explode('.', $qualified, 2);
            $path = $module . '/' . $name;
            $path = $this->appendTailToPath($path, $tail);
            return $this->request->url($path);
        }

        $viewWord = $this->settings->get(ConfigKeys::PAGE_PARAM, 'view');
        $qs = $viewWord . '=' . rawurlencode($qualified);
        $qs = $this->appendTailToQuery($qs, $tail);
        return $this->request->url('?' . $qs);
    }

    /**
     * @param string $qualified "Module.action"
     * @param array|string|null $tail
     * @return string
     */
    public function action($qualified, $tail = null)
    {
        $qualified = (string)$qualified;

        if ($this->enabled()) {
            list($module, $name) = explode('.', $qualified, 2);
            $path = 'api/' . $module . '/' . $name;
            $path = $this->appendTailToPath($path, $tail);
            return $this->request->url($path);
        }

        $actionWord = $this->settings->get(ConfigKeys::ACTION_PARAM, 'action');
        $qs = $actionWord . '=' . rawurlencode($qualified);
        $qs = $this->appendTailToQuery($qs, $tail);
        return $this->request->url('?' . $qs);
    }

    public function qualifyView($module, $viewName)
    {
        return (string)$module . '.' . (string)$viewName;
    }

    public function qualifyAction($module, $actionName)
    {
        return (string)$module . '.' . (string)$actionName;
    }

    // -------------------------
    // Internals
    // -------------------------

    private function appendTailToPath($path, $tail)
    {
        $segments = $this->normalizeTail($tail);
        if (!$segments) return (string)$path;

        $out = (string)$path;
        foreach ($segments as $seg) {
            $out .= '/' . rawurlencode((string)$seg);
        }
        return $out;
    }

    private function appendTailToQuery($qs, $tail)
    {
        $segments = $this->normalizeTail($tail);
        if (!$segments) return (string)$qs;

        // Use tail[]= to preserve ordering and avoid delimiter ambiguity
        $out = (string)$qs;
        foreach ($segments as $seg) {
            $out .= '&tail[]=' . rawurlencode((string)$seg);
        }
        return $out;
    }

    /**
     * @param array|string|null $tail
     * @return array
     */
    private function normalizeTail($tail)
    {
        if ($tail === null) return array();

        if (!is_array($tail)) {
            $s = trim((string)$tail);
            if ($s === '') return array();

            // allow "a/b/c" convenience in PHP usage (optional)
            if (strpos($s, '/') !== false) {
                $parts = explode('/', $s);
                $out = array();
                foreach ($parts as $p) {
                    $p = trim((string)$p);
                    if ($p !== '') $out[] = $p;
                }
                return $out;
            }

            return array($s);
        }

        $out = array();
        foreach ($tail as $v) {
            if ($v === null) continue;
            $s = trim((string)$v);
            if ($s === '') continue;
            $out[] = $s;
        }
        return $out;
    }
}
