<?php
namespace boru\dweb\Routing;

use boru\dweb\Config\ConfigKeys;
use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Http\Request;
use boru\dweb\Http\NotFoundException;

class PathRouter
{
    /** @var SettingsInterface */
    private $config;

    /** @var RouteCollection */
    private $routes;

    /** @var WebRouter */
    private $fallback;

    public function __construct(SettingsInterface $config, RouteCollection $routes, WebRouter $fallback)
    {
        $this->config = $config;
        $this->routes = $routes;
        $this->fallback = $fallback;
    }

    public function dispatch(Request $req)
    {
        $path = $req->path();
        $base = $this->detectBasePath($req);
        $path = $this->stripBasePath($path, $base);
        $path = trim($path, '/');

        if ($path === '') {
            return $this->dispatchDefault($req);
        }

        if ((bool)$this->config->get(ConfigKeys::DEBUG_ENABLED, false) && (bool)$this->config->get(ConfigKeys::DEBUG_ROUTES_ENABLED, true)) {
            $dumpPath = (string)$this->config->get(ConfigKeys::DEBUG_ROUTES_PATH, '/__dweb/routes');
            if ($req->path() === $dumpPath) {
                $payload = \boru\dweb\Debug\RouteDump::toArray($this->routes);
                return new \boru\dweb\Http\JsonResponse($payload, 200);
            }
        }

        $parts = explode('/', $path);

        // /api/<Module>/<Action>/<tail...>
        if (isset($parts[0]) && strtolower($parts[0]) === 'api') {
            if (!isset($parts[1]) || $parts[1] === '') throw new NotFoundException('Missing module in /api route');
            if (!isset($parts[2]) || $parts[2] === '') throw new NotFoundException('Missing action in /api route');

            $module = $parts[1];
            $action = $parts[2];
            $tail   = array_slice($parts, 3);

            $qualified = $module . '.' . $action;

            $actionParam = $this->config->get(ConfigKeys::ACTION_PARAM, 'action');
            $req2 = $req->withQueryParams(array($actionParam => $qualified));

            $record = $this->routes->getAction($qualified);
            if ($record && $this->actionAllowsMethod($record, $req2->method())) {

                $schema = isset($record['pathParams']) ? $record['pathParams'] : null;
                $req2 = $this->attachPathParams($req2, $qualified, $tail, $schema);

                return call_user_func($record['handler'], $req2);
            }

            if ($record) {
                throw new \boru\dweb\Http\HttpException(405, 'Method Not Allowed', 'Method not allowed for action: ' . $qualified);
            }
            throw new NotFoundException('Action not found: ' . $qualified);
        }

        // /<Module>/<View>/<tail...>
        $defaultView = $this->config->get(ConfigKeys::DEFAULT_VIEW, '');
        $module = $parts[0];
        $viewName = isset($parts[1]) && $parts[1] !== '' ? $parts[1] : $defaultView;
        $tail = array_slice($parts, 2);

        $qualified = $module . '.' . $viewName;

        $pageParam = $this->config->get(ConfigKeys::PAGE_PARAM, 'view');
        $req2 = $req->withQueryParams(array($pageParam => $qualified));

        $rec = $this->routes->getViewRecord($qualified);
        if ($rec) {
            $schema = isset($rec['pathParams']) ? $rec['pathParams'] : null;
            $req2 = $this->attachPathParams($req2, $qualified, $tail, $schema);

            return call_user_func($rec['handler'], $req2);
        }

        return $this->fallback->dispatch($req);
    }

    private function dispatchDefault(Request $req)
    {
        if (!$this->config->get(ConfigKeys::DEFAULT_MODULE, null)) {
            return $this->fallback->dispatch($req);
        }

        $qualified = $this->config->get(ConfigKeys::DEFAULT_MODULE) . '.' . $this->config->get(ConfigKeys::DEFAULT_VIEW, '');
        $pageParam = $this->config->get(ConfigKeys::PAGE_PARAM, 'view');
        $req2 = $req->withQueryParams(array($pageParam => $qualified));

        $rec = $this->routes->getViewRecord($qualified);
        if ($rec) {
            // default has no tail
            $req2 = $this->attachPathParams($req2, $qualified, array(), isset($rec['pathParams']) ? $rec['pathParams'] : null);
            return call_user_func($rec['handler'], $req2);
        }

        throw new NotFoundException('Default view not found: ' . $qualified);
    }

    private function attachPathParams(Request $req, $qualified, array $tail, $schema)
    {
        $named = array();
        if (is_array($schema)) {
            $i = 0;
            foreach ($schema as $key) {
                if (!isset($tail[$i])) break;
                $named[(string)$key] = $tail[$i];
                $i++;
            }
        }

        // requires Request::withMeta() (see below)
        return $req
            ->withMeta('dweb.route', (string)$qualified)
            ->withMeta('dweb.path.tail', $tail)
            ->withMeta('dweb.path.params', $named);
    }

    private function actionAllowsMethod($actionRecord, $method)
    {
        if (!$actionRecord) return false;
        if (!isset($actionRecord['methods']) || $actionRecord['methods'] === null) return true;
        return isset($actionRecord['methods'][strtoupper((string)$method)]);
    }

    private function detectBasePath(Request $req)
    {
        $base = $this->config->get(ConfigKeys::BASE_PATH, null);
        if ($base !== null) {
            $base = (string)$base;
            if ($base === '' || $base === '/') return '';
            if ($base[0] !== '/') $base = '/' . $base;
            return rtrim($base, '/');
        }

        $script = (string)$req->server('SCRIPT_NAME', '');
        if ($script === '') return '';

        $dir = str_replace('\\', '/', dirname($script));
        if ($dir === '.' || $dir === '/' || $dir === '\\') return '';
        return rtrim($dir, '/');
    }

    private function stripBasePath($uriPath, $base)
    {
        $uriPath = (string)$uriPath;
        $base    = (string)$base;

        if ($base === '') return $uriPath;

        if ($uriPath === $base) return '/';

        if (strpos($uriPath, $base . '/') === 0) {
            $rest = substr($uriPath, strlen($base));
            return $rest !== '' ? $rest : '/';
        }

        return $uriPath;
    }
}
