<?php
namespace boru\dweb\Support;

use boru\dweb\DwebConfig;
use boru\dweb\WebUI;
use boru\dweb\Config\ConfigKeys;

/**
 * Shared boot entry for:
 * - host app public/index.php
 * - tests
 * - vendor/bin scripts
 *
 * Uses KernelBuilder as single wiring source of truth.
 */
final class Bootstrap
{
    private function __construct() {}

    /**
     * @return string absolute package root (vendor/bin safe)
     */
    public static function packageRoot()
    {
        $rc = new \ReflectionClass(__CLASS__);
        $file = $rc->getFileName(); // .../src/Support/Bootstrap.php
        return dirname(dirname(dirname($file)));
    }

    /**
     * Build a default WebConfig + apply overrides.
     *
     * @param array $opts
     * @param callable|null $configure function(WebConfig $config): void
     * @return DwebConfig
     */
    public static function config(array $opts = array(), $configure = null)
    {
        $config = DwebConfig::createDefault();

        // keep your deterministic defaults style, but allow caller to override
        if (!isset($opts[ConfigKeys::DEBUG_ENABLED]))        $config->set(ConfigKeys::DEBUG_ENABLED, true);
        if (!isset($opts[ConfigKeys::DEBUG_ROUTES_ENABLED])) $config->set(ConfigKeys::DEBUG_ROUTES_ENABLED, true);

        // Deterministic defaults (copied from your build_env)
        $config->set(ConfigKeys::DEFAULT_MODULE, isset($opts['defaultModule']) ? $opts['defaultModule'] : 'Skeleton');
        $config->set(ConfigKeys::DEFAULT_VIEW, isset($opts['defaultPage']) ? $opts['defaultPage'] : 'home');
        $config->set(ConfigKeys::PAGE_PARAM, isset($opts['pageParam']) ? $opts['pageParam'] : 'view');
        $config->set(ConfigKeys::ACTION_PARAM, isset($opts['actionParam']) ? $opts['actionParam'] : 'action');

        // Security headers ON by default (copied)
        $config->set(ConfigKeys::SECURITY_HEADERS_ENABLED, true);
        $config->set(ConfigKeys::SECURITY_HEADERS_CONTENT_SECURITY_POLICY, null);
        $config->set(ConfigKeys::SECURITY_HEADERS_X_FRAME_OPTIONS, null);
        $config->set(ConfigKeys::SECURITY_HEADERS_X_CONTENT_TYPE_OPTIONS, null);
        $config->set(ConfigKeys::SECURITY_HEADERS_X_XSS_PROTECTION, null);

        // Apply key=>value overrides (ConfigKeys etc.)
        foreach ($opts as $k => $v) {
            // skip the special non-config options above
            if ($k === 'defaultModule' || $k === 'defaultPage' || $k === 'pageParam' || $k === 'actionParam') {
                continue;
            }
            $config->set((string)$k, $v);
        }

        if ($configure) {
            call_user_func($configure, $config);
        }

        return $config;
    }

    /**
     * Build core kernel pieces for tests / scripts.
     *
     * @param array $opts
     * @param callable|null $configureContainer function(Container $c, WebConfig $config): void
     * @param callable|null $configureModules function(ModuleManager $mm, Container $c, WebConfig $config): void
     * @param array|null $middleware
     *
     * @return array [WebConfig $config, Container $container, $routes, callable $pipeline]
     */
    public static function envO(array $opts = array(), $configureContainer = null, $configureModules = null, array $middleware = null)
    {
        $config = self::config($opts);

        list($container, $routes, $router, $pipeline) = KernelBuilder::build(
            $config,
            $configureContainer,
            $configureModules,
            $middleware
        );

        return array($config, $container, $routes, $pipeline);
    }

    public static function envObject(array $opts = array(), $configureContainer = null, $configureModules = null, array $middleware = null)
    {
        $config = self::config($opts);

        $spec = \boru\dweb\Support\KernelSpec::fromConfig($config)
            ->withContainer($configureContainer)
            ->withModules($configureModules)
            ->withMiddleware($middleware);

        return \boru\dweb\Support\KernelBuilder::buildEnv($spec);
    }


    /**
     * Build env using a host bootstrap file.
     *
     * Bootstrap file must return HostBootstrap.
     *
     * @param string $bootstrapPath
     * @param array $overrides overrides/extra opts (merged on top of file opts)
     * @param array|null $middlewareOverride if non-null, wins over bootstrap middleware
     *
     * @return array [WebConfig $config, Container $container, $routes, callable $pipeline]
     */
    public static function envFromFile($bootstrapPath, array $overrides = array(), array $middlewareOverride = null)
    {
        $host = \boru\dweb\Support\BootstrapFile::load($bootstrapPath);

        // Merge opts: file opts + overrides (overrides win)
        $opts = $host->opts();
        foreach ($overrides as $k => $v) {
            $opts[$k] = $v;
        }
        $middleware = $middlewareOverride !== null ? $middlewareOverride : $host->middleware();

        return self::envObject(
            $opts,
            $host->configureContainer(),
            $host->configureModules(),
            $middleware
        );
    }


    /**
     * Web runner for host apps.
     *
     * NOTE: WebUI likely already has logic to build container/router internally.
     * The recommended direction is to refactor WebUI to call KernelBuilder::build()
     * so WebUI, tests, and vendor/bin share identical wiring.
     *
     * @param array $opts
     * @param callable|null $configure function(WebConfig $config, WebUI $ui): void
     * @return void
     */
    public static function run(array $opts = array(), $configure = null)
    {
        $config = self::config($opts);

        $ui = new WebUI($config);

        if ($configure) {
            call_user_func($configure, $config, $ui);
        }

        $ui->render();
    }
}
