<?php
namespace boru\dweb\Support;

use boru\dweb\DwebConfig;
use boru\dweb\Kernel\Container;
use boru\dweb\Kernel\ModuleManager;

use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Contracts\RendererInterface;
use boru\dweb\Contracts\TemplateLocatorInterface;
use boru\dweb\Contracts\LoggerInterface;
use boru\dweb\Assets\AssetPublisherInterface;

use boru\dweb\Rendering\TemplateLocator;
use boru\dweb\Rendering\SmartyRenderer;

use boru\dweb\Logging\NullLogger;
use boru\dweb\Http\Request;

use boru\dweb\Routing\WebRouter;
use boru\dweb\Routing\PathRouter;
use boru\dweb\Routing\CanonicalUrl;

use boru\dweb\Middleware\MethodOverrideMiddleware;
use boru\dweb\Middleware\JsonBodyMiddleware;
use boru\dweb\Middleware\SecurityHeadersMiddleware;

use boru\dweb\Assets\FilesystemAssetPublisher;


/**
 * Builds core runtime pieces (container, routes, routers, middleware pipeline).
 *
 * This is the single source of truth for wiring in:
 * - WebUI (host apps)
 * - CLI smoke tests
 * - vendor/bin scripts
 *
 * PHP 5.6 compatible.
 */
final class KernelBuilder
{
    private function __construct() {}

    /**
     * @param DwebConfig $config
     * @param callable|null $configureContainer function(Container $c, WebConfig $config): void
     * @param callable|null $configureModules function(ModuleManager $mm, Container $c, WebConfig $config): void
     * @param array|null $middleware Optional override list of middleware instances.
     *
     * @return array [Container $container, $routes, $router, callable $pipeline]
     */
    public static function build(
        DwebConfig $config,
        $configureContainer = null,
        $configureModules = null,
        array $middleware = null
    ) {
        $container = new Container();

        // Core singletons
        $container->set(SettingsInterface::class, $config);

        // Logger default (host can override via configureContainer)
        if (!$container->has(LoggerInterface::class)) {
            $container->set(LoggerInterface::class, new NullLogger());
        }

        // Template locator
        $locator = new TemplateLocator($config);
        $container->set(TemplateLocatorInterface::class, $locator);

        // Request factory (works in Web + CLI if caller chooses to use make_req for tests)
        $container->factory(
            Request::class,
            function () {
                return Request::fromGlobals();
            }
        );

        // CanonicalUrl (depends on Settings + Request)
        $container->factory(
            CanonicalUrl::class,
            function ($c) {
                return new CanonicalUrl(
                    $c->get(SettingsInterface::class),
                    $c->get(Request::class)
                );
            }
        );

        // Renderer
        $container->set(RendererInterface::class, function ($c) {
            $settings  = $c->get(SettingsInterface::class);
            $locator   = $c->get(TemplateLocatorInterface::class);
            $canonical = $c->get(CanonicalUrl::class);
            $smarty    = SmartyRenderer::buildSmarty($settings);

            return new SmartyRenderer($smarty, $locator, $settings, $canonical);
        });

        // Host hook: add/override services
        if ($configureContainer) {
            call_user_func($configureContainer, $container, $config);
        }

        $container->set(\boru\dweb\Mvc\ModuleNamespaceResolver::class, function () {
            return new \boru\dweb\Mvc\ModuleNamespaceResolver();
        });

        // Modules + routes
        $modules = new ModuleManager();
        if ($configureModules) {
            call_user_func($configureModules, $modules, $container, $config);
        }
        // ✅ Make ModuleManager discoverable by factories/handlers
        $container->set(\boru\dweb\Kernel\ModuleManager::class, $modules);
        $container->label(\boru\dweb\Kernel\ModuleManager::class, 'ModuleManager');

        // Smoke suite registry
        $container->set(\boru\dweb\Smoke\SmokeSuiteRegistry::class, function () {
            return new \boru\dweb\Smoke\SmokeSuiteRegistry();
        });

        // CLI command registry
        $container->set(\boru\dweb\Cli\CommandRegistry::class, function () {
            return new \boru\dweb\Cli\CommandRegistry();
        });

        // Allow modules to register suites/commands (optional)
        $container->set(\boru\dweb\Support\ModuleExtensions::class, function ($c) use ($modules) {
            return new \boru\dweb\Support\ModuleExtensions($modules, $c);
        });

        // Collect module-provided suites/commands (explicit opt-in)
        $ext = $container->get(\boru\dweb\Support\ModuleExtensions::class);
        $ext->registerSuites();
        $ext->registerCommands();


        // If host didn't add modules, that's allowed; routes may be empty.
        $routes = $modules->boot($container, $config);

        // Asset publishing (optional capability; safe default)
        $container->set(AssetPublisherInterface::class, function ($c) use ($config, $modules) {
            return new FilesystemAssetPublisher(
                $c->get(SettingsInterface::class),
                $modules,
                $c->get(LoggerInterface::class)
            );
        });

        // Routers
        $queryRouter = new WebRouter($config, $routes);
        $router = new PathRouter($config, $routes, $queryRouter);

        // Middleware pipeline (default order matches your CLI build_env)
        if ($middleware === null) {
            $middleware = array(
                new MethodOverrideMiddleware(),
                new JsonBodyMiddleware($container->get(LoggerInterface::class)),
                new SecurityHeadersMiddleware($config),
            );
        }

        $dispatch = function ($request) use ($router) {
            return $router->dispatch($request);
        };

        $pipeline = $dispatch;
        for ($i = count($middleware) - 1; $i >= 0; $i--) {
            $mw = $middleware[$i];
            $next = $pipeline;
            $pipeline = function ($request) use ($mw, $next) {
                return $mw->handle($request, $next);
            };
        }

        return array($container, $routes, $router, $pipeline);
    }

    public static function buildEnv(\boru\dweb\Support\KernelSpec $spec)
    {
        list($container, $routes, $router, $pipeline) = self::build(
            $spec->config(),
            $spec->configureContainer(),
            $spec->configureModules(),
            $spec->middleware()
        );

        return new \boru\dweb\Support\KernelEnv(
            $spec->config(),
            $container,
            $routes,
            $router,
            $pipeline
        );
    }

}
