# Rendering in boru/dweb

Rendering in `boru/dweb` is **explicit, layered, and contract-driven**.

There is no implicit controller rendering, no auto-mapped templates,
and no framework-owned UI.

---

## Core Principles

- Controllers explicitly return responses
- Rendering is not tied to routing
- Templates are module-scoped
- Layouts are explicit
- Fragments are first-class
- Rendering engines are swappable

---

## Controllers: Views and Actions

Both **Views** and **Actions**:

- may render templates
- may return fragments
- may return JSON
- may return SSE streams

The distinction is **semantic**, not technical.

| Type | Typical use |
|----|----|
| View | Pages, fragments, layout composition |
| Action | APIs, form handlers, SSE streams |

---

## Rendering Flow

```

Controller
→ Renderer
→ TemplateLocator
→ Template
→ Layout (optional)
→ Response

```

Nothing is inferred automatically.

---

## Rendering a Full Page

```php
return $this->render(
    'layouts/default',
    'home/index',
    ['title' => 'Hello']
);
```

---

## Fragments (Layout-Free Rendering)

Fragments are templates rendered **without layouts**.

```php
return $this->fragment('partials/item', ['item' => $item]);
```

Fragments are ideal for:

* HTMX
* AJAX
* SSE
* progressive enhancement

---

## Sections and Layout Composition

Controllers may populate named sections explicitly.

```php
$this->sectionSet('sidebar',
    $this->renderPartialToString('partials/sidebar')
);
```

Layouts render sections explicitly:

```smarty
<aside id="{$dweb->sectionId('sidebar')}">
  {$sections.sidebar|default:'' nofilter}
</aside>

<main id="{$dweb->sectionId('content')}">
  {$sections.content|default:$content nofilter}
</main>
```

There is:

* no output buffering
* no implicit capture
* no magic slots

---

## Fragment → Section Targeting

Fragments may declare which section they update:

```php
return $this->fragmentToSection(
    'sidebar',
    'partials/sidebar',
    ['items' => $items]
);
```

Response headers indicate intent:

```
X-DWeb-Fragment: Skeleton.sidebar
X-DWeb-Section: sidebar
```

---

## Server-Sent Events (SSE)

dweb supports **native SSE streaming** via `SseResponse`.

SSE responses are first-class and integrate with the standard response emitter.

### Streaming SSE from an Action

```php
return SseResponse::stream(function ($send) {
    $send('<div>Starting...</div>', 'status');

    for ($i = 1; $i <= 5; $i++) {
        $send('<span>' . $i . '</span>', 'content');
        sleep(1);
    }

    $send('<div>Done</div>', 'status');
});
```

SSE is ideal for:

* live updates
* progress indicators
* streaming AI output
* HTMX SSE integration

---

## HTMX + SSE

HTMX can consume SSE directly using its `sse` extension.

dweb SSE streams typically emit **HTML fragments**, not JSON,
allowing server-rendered UI updates with zero client JS.

---

## Renderer Implementations

The framework does not mandate a renderer.

Current implementation:

* Smarty (via shim)

Possible future renderers:

* Native PHP
* Twig
* Static pre-rendering

Controllers do not change when renderers change.

Views and Actions access URL helpers via `$this->dweb()`, provided by `ControllerContext`.

---

## Design Rules

* Controllers control responses
* Renderers control rendering
* Templates contain presentation only
* No auto-mapping of templates
* No hidden behavior

If something renders, it should be obvious where it came from.