# 🔁 boru/dweb — Agent Primer (Current State, Post–Fun Layers Phase 1)

This document is a **handoff primer** for continuing work on the `boru/dweb` framework.  
All architectural decisions below are **final unless explicitly stated otherwise**.

The next agent should assume this context as ground truth.

---

## 1. What This Project Is

`boru/dweb` is a **lightweight, explicit PHP web framework**.

It is:

- A **framework**, not an application
- Intentionally **non-magical**
- Designed for **clarity, debuggability, and long-term maintainability**
- Compatible with **PHP 5.6 → PHP 8.x**

There are **no backward-compatibility requirements** at this stage. We are designing intentionally.

---

## 2. PHP Compatibility (LOCKED IN)

The framework must remain compatible with:

- PHP 5.6 (minimum)
- PHP 7.x
- PHP 8.x

Implications:

- ❌ No typed properties
- ❌ No union types
- ❌ No arrow functions
- ❌ No constructor property promotion
- ❌ No attributes
- ❌ No reliance on PHP 7+ stdlib features without fallbacks

---

## 3. Core Architectural Principles (LOCKED IN)

### Core vs Modules

**Core owns ZERO application endpoints.**

Core responsibilities:

- HTTP Request / Response
- Routing
- Middleware pipeline
- DI container
- Module lifecycle
- Rendering contracts
- Error handling
- Debug tooling
- WebUI orchestration

Modules responsibilities:

- Views (HTML)
- Actions (JSON / SSE / side effects)
- Domain services
- Templates
- Assets
- Integration logic

---

## 4. Terminology (LOCKED IN)

| Term   | Meaning                                        |
|--------|------------------------------------------------|
| View   | Controller returning HTML                      |
| Action | Controller returning JSON / SSE                |
| Module | Feature bundle (routes + services + templates) |

There are **no Pages**.  
Everything HTML is a **View**.

---

## 5. Namespacing (LOCKED IN)

Core:

```php
boru\dweb*
````

Modules:

```php
boru\dweb\Modules\<ModuleName>*
```

Example:

```php
boru\dweb\Modules\Skeleton\Views\HomeView
boru\dweb\Modules\Skeleton\Actions\EchoAction
```

---

## 6. Routing Model (LOCKED IN)

Routes are always qualified:

```text
<Module>.<name>
```

Supported access patterns:

* `/Skeleton/home`
* `/api/Skeleton/echo`
* `/?view=Skeleton.home`
* `/?action=Skeleton.echo`

Default module + view are configurable.

Routing is deterministic and case-preserving.

---

## 7. Dependency Injection (LOCKED IN)

* Explicit container
* No reflection
* No autowiring
* No magic

Container supports:

```php
set($id, $value)
set($id, callable)        // shared
factory($id, callable)    // non-shared
get($id)
```

### Services Helper (DX Layer)

A `Services` helper exists for typed, IDE-friendly access:

```php
Services::renderer($container);
Services::canonicalUrl($container);
Services::request($container);
```

This is **pure sugar** over `$container->get(Foo::class)`.

---

## 8. Settings / Configuration (LOCKED IN)

`WebConfig` is **config-only**.

* No public properties
* No service storage
* No container behavior

All config access uses **ConfigKeys** constants for IDE autocomplete:

```php
ConfigKeys::DEBUG_ENABLED
ConfigKeys::URLS_CANONICAL_ENABLED
ConfigKeys::ASSET_URL_PREFIX
```

`SettingsInterface` === configuration only.

---

## 9. Rendering System (LOCKED IN)

Rendering is:

* Explicit
* Contract-driven
* Layout-aware
* Override-friendly

Core contracts:

* `RendererInterface`
* `TemplateLocatorInterface`

Current renderer:

* Smarty (via shim)

Views explicitly call render methods. Nothing is inferred.

---

## 10. Fun Layers (IMPLEMENTED)

Fun layers are **optional capability extensions**.
They do not modify routing, DI, or module contracts.

### ✅ Implemented Fun Layers

#### Fragment Rendering

* Layout-free views
* Tagged responses
* Client-agnostic (HTMX, JS, SSE)

Headers include:

```text
X-DWeb-Fragment
```

---

#### SectionBag (Layout Composition)

* Explicit named sections
* No output buffering
* No implicit capture
* Sections are HTML strings

Views populate sections explicitly:

```php
$this->sectionSet('sidebar', $html);
$this->sectionAppend('scripts', $js);
```

Layouts render sections explicitly.

---

#### Fragment → Section Targeting

Fragments may declare **which section they update**.

```php
return $this->fragmentToSection(
    'sidebar',
    'sidebar',
    $data
);
```

Response metadata:

```text
X-DWeb-Section
X-DWeb-Selector
X-DWeb-Swap
```

---

#### Section Selector Convention

Default convention:

```text
section name: sidebar
DOM id:        section-sidebar
selector:      #section-sidebar
```

Helpers available in templates:

```smarty
{$dweb->sectionId('sidebar')}
{$dweb->sectionSelector('sidebar')}
```

---

#### HTMX Response Helpers

* Explicit HX headers
* No auto-detection
* Safe for non-HTMX clients

---

#### Canonical URL Helpers

* Config-driven canonical vs query URLs
* Base-path aware
* Template helpers:

```smarty
{$dweb->view('home')}
{$dweb->action('echo')}
```

---

#### AssetManager (Section-Aware)

* Assets are registered into sections
* No global state
* No buffering
* Layout controls placement

---

## 11. Error Handling & DX

* Typed exceptions preferred over `RuntimeException`
* Context-carrying exceptions exist
* Error rendering handled centrally
* Stack traces are being actively improved for sanity/readability

---

## 12. Debug Tooling (LOCKED IN)

When debug is enabled:

```text
GET /__dweb/routes
```

Returns JSON of:

* all views
* all actions
* allowed methods

This endpoint:

* is router-level
* is not a module
* works in CLI + WebUI

---

## 13. Testing Philosophy (LOCKED IN)

Primary testing:

* CLI smoke tests

Test layers:

```text
Router-level smoke tests
WebUI end-to-end smoke tests
```

No PHPUnit requirement.

---

## 14. Current Docs State

Docs included in the provided zip:

* `01_overview.md`
* `02_module_authoring.md`
* `03_rendering.md` (updated with fragments + sections)
* `04_middleware.md`
* `05_error_handling.md`
* `06_testing.md`
* `07_fun_layers.md` (updated)

---

## 15. What the NEXT Agent Should Do

The foundation is **stable**.

The next agent should:

* ❌ NOT redesign core architecture
* ❌ NOT add framework magic
* ❌ NOT change existing contracts

The next agent MAY:

* Add new **fun layers**
* Improve DX (debug output, tooling)
* Extend UI composition helpers
* Add SSE helpers
* Add auth / identity middleware patterns
* Improve documentation
* Refine error reporting

All extensions should be **explicit**, **opt-in**, and **client-agnostic**.

---

## Final Instruction to Agent

Assume all decisions above are final.

Extend capabilities, not architecture.

If something feels unclear, prefer:

* explicit APIs
* small helpers
* additional layers

Nothing should “just happen.”

```