<?php
namespace boru\dweb\Cli;

use boru\dweb\Support\KernelEnv;
use boru\dweb\Cli\Commands\HelpCommand;
use boru\dweb\Cli\Commands\PublishCommand;
use boru\dweb\Cli\Commands\SmokeTestCommand;

class App
{
    /** @var CommandRegistry */
    private $registry;

    public function __construct(CommandRegistry $registry = null)
    {
        $this->registry = $registry ? $registry : new CommandRegistry();
    }

    /**
     * @param array $argv
     * @return int
     */
    public function run(array $argv)
    {
        $tokens = array_slice($argv, 1);

        // Pass 1: extract env flags anywhere in argv
        list($envArgs, $rest) = Args::extractEnv($tokens);

        // Determine command name from remaining tokens:
        // first non-flag token, otherwise "help"
        $cmdName = 'help';
        $cmdIndex = null;

        for ($i = 0; $i < count($rest); $i++) {
            $t = $rest[$i];

            // global help shortcut
            if ($t === '-h' || $t === '--help') {
                $cmdName = 'help';
                $cmdIndex = $i;
                break;
            }

            // first non-flag token is command name
            if (strlen($t) > 0 && $t[0] !== '-') {
                $cmdName = $t;
                $cmdIndex = $i;
                break;
            }
        }

        // args after the command token
        $cmdArgs = array();
        if ($cmdIndex !== null) {
            $cmdArgs = array_slice($rest, $cmdIndex + 1);
        } else {
            // no command token found; allow "help" with remaining flags
            $cmdArgs = $rest;
        }

        // Build env BEFORE registry if:
        // - command isn't "help", OR
        // - env flags were provided (so help can show module commands)
        $env = null;
        if ($cmdName !== 'help' || !$envArgs->isEmpty()) {
            try {
                $env = CliEnv::build($envArgs);
            } catch (\Exception $e) {
                $this->err("ERROR: " . $e->getMessage() . "\n\n");
                $this->err("Tip: dweb help\n");
                return 2;
            }
        }

        // Build registry with env-aware commands
        $this->registry = $this->buildRegistry($env);

        // Allow env/modules to extend registry (commands provided by modules)
        if ($env) {
            $this->maybeExtendRegistryFromEnv($env);
        }

        // Resolve command
        $command = $this->registry->get($cmdName);
        if (!$command) {
            $this->err("Unknown command: {$cmdName}\n\n");
            $this->registry->get('help')->execute(array());
            return 2;
        }

        // Execute (CommandInterface remains execute(array $args))
        return $command->execute($cmdArgs);
    }

    /**
     * @param KernelEnv|null $env
     * @return CommandRegistry
     */
    private function buildRegistry($env)
    {
        $r = new CommandRegistry();

        $r->add(new HelpCommand($this));

        // Core commands can self-error if env is null (help still works)
        $r->add(new SmokeTestCommand($env));
        $r->add(new PublishCommand($env));

        return $r;
    }

    /**
     * Optional: let host env register commands into a registry service.
     *
     * Pattern:
     *   container->set(CommandRegistry::class, $registry)
     *   modules add commands during register()/commands()
     */
    private function maybeExtendRegistryFromEnv(KernelEnv $env)
    {
        try {
            $c = $env->container();

            $extra = $c->get(\boru\dweb\Cli\CommandRegistry::class);
            if ($extra instanceof \boru\dweb\Cli\CommandRegistry) {
                foreach ($extra->all() as $cmd) {
                    $this->registry->add($cmd);
                }
            }
        } catch (\Exception $ignore) {
            // no-op
        }
    }

    /** @return CommandInterface[] */
    public function listCommands()
    {
        return $this->registry->all();
    }

    public function out($s) { echo $s; }
    public function err($s) { fwrite(STDERR, $s); }
}
