<?php
namespace boru\dweb;

use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Config\ConfigKeys;

class DwebConfig implements SettingsInterface
{
    /** @var array */
    private $settings = array();

    /**
     * SettingsInterface
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function get($key, $default = null)
    {
        $k = (string)$key;
        return array_key_exists($k, $this->settings) ? $this->settings[$k] : $default;
    }

    /**
     * SettingsInterface
     * @param string $key
     * @param mixed $value
     * @return $this
     */
    public function set($key, $value)
    {
        $this->settings[(string)$key] = $value;
        return $this;
    }

    /**
     * SettingsInterface
     * @return array
     */
    public function all()
    {
        return $this->settings;
    }

    public static function createDefault()
    {
        $config = new self();

        // Infer mount path from URL (e.g. /boruaiweb/index.php => /boruaiweb/)
        $scriptName = isset($_SERVER['SCRIPT_NAME']) ? (string)$_SERVER['SCRIPT_NAME'] : '';
        $mount = '/';
        if ($scriptName !== '') {
            $mount = rtrim(str_replace('\\', '/', dirname($scriptName)), '/');
            if ($mount === '' || $mount === '.') $mount = '/';
            $mount = rtrim($mount, '/') . '/';
        }

        // Host override templates root (optional)
        // Structure: <root>/<Module>/<templateId>.tpl
        $config->set(ConfigKeys::TEMPLATES_PATH, null);

        // Smarty defaults
        $config->set(ConfigKeys::SMARTY_COMPILE_DIR, sys_get_temp_dir() . '/dweb_smarty/compile');
        $config->set(ConfigKeys::SMARTY_CACHE_DIR,   sys_get_temp_dir() . '/dweb_smarty/cache');
        $config->set(ConfigKeys::SMARTY_COMPILE_CHECK, true);
        $config->set(ConfigKeys::SMARTY_CACHING, false);
        $config->set(ConfigKeys::SMARTY_CACHE_LIFETIME, 0);

        // Logging
        $config->set(ConfigKeys::LOGGER_CLASS, null); // host may inject a logger instance

        // Security headers (middleware)
        $config->set(ConfigKeys::SECURITY_HEADERS_ENABLED, true);
        $config->set(ConfigKeys::SECURITY_HEADERS_CONTENT_SECURITY_POLICY, null); // e.g. "default-src 'self'; img-src 'self' data:;"
        // Sane defaults
        $config->set(ConfigKeys::URLS_CANONICAL_ENABLED, true);

        // Debug route dump
        $config->set(ConfigKeys::DEBUG_ROUTES_ENABLED, true);
        $config->set(ConfigKeys::DEBUG_ROUTES_PATH, '/__dweb/routes');

        $config->set(ConfigKeys::ASSET_URL_PREFIX, $mount);   // where $mount ends with '/'
        $config->set(ConfigKeys::ASSET_VERSION, null);        // optional cache busting
        $config->set(ConfigKeys::ASSET_PUBLISH_ENABLED, false);
        $config->set(ConfigKeys::ASSET_PUBLIC_DIR, '');

        //find bin/dweb relative to current working dir
        $binPath = 'vendor/bin/dweb';
        if (is_file(getcwd() . '/' . $binPath)) {
            $config->set(ConfigKeys::CLI_BIN_PATH, $binPath);
        } elseif(is_file(getcwd() . '/../../' . $binPath)) {
            $config->set(ConfigKeys::CLI_BIN_PATH, '../../' . $binPath);
        } else {
            $config->set(ConfigKeys::CLI_BIN_PATH, $binPath); //fallback, may not work
        }
        $config->set(\boru\dweb\Config\ConfigKeys::CLI_ENABLED, true);
        $config->set(\boru\dweb\Config\ConfigKeys::CLI_CWD, getcwd());
        $config->set(\boru\dweb\Config\ConfigKeys::CLI_ENV_BOOTSTRAP, null);
        $config->set(\boru\dweb\Config\ConfigKeys::CLI_TIMEOUT_SEC, null);

        return $config;
    }


    public function withAssetUrlPrefix($prefix) { $this->set(ConfigKeys::ASSET_URL_PREFIX, $prefix); return $this; }
    public function withDefaultModule($moduleName) { $this->set(ConfigKeys::DEFAULT_MODULE, $moduleName); return $this; }
    public function withTemplatesPath($path) { $this->set(ConfigKeys::TEMPLATES_PATH, $path); return $this; }
    public function withCanonicalUrls($enabled) { $this->set(ConfigKeys::URLS_CANONICAL_ENABLED, (bool)$enabled); return $this; }
    /**
     * Optional helper if you want to set many at once.
     * @param array $settings
     * @return $this
     */
    public function withSettings(array $settings)
    {
        foreach ($settings as $k => $v) {
            $this->settings[(string)$k] = $v;
        }
        return $this;
    }
}
