<?php
namespace boru\dweb\Http;

use boru\dweb\Config\ConfigKeys;
use boru\dweb\Exceptions\ContextException;
use boru\dweb\Http\HttpException;

class ExceptionResponder
{
    public static function fromException($e, Request $req = null, $debug = false, $settings = null)
    {
        $status = self::statusFromException($e);
        $exceptionMessage = self::safeMessage($e);

        $sanitizeUser  = $settings ? (bool)$settings->get(ConfigKeys::ERRORS_SANITIZE_USER_MESSAGES, true) : true;
        $sanitizeDebug = $settings ? (bool)$settings->get(ConfigKeys::ERRORS_SANITIZE_DEBUG_DETAILS, false) : false;
        $showTrace     = $settings ? (bool)$settings->get(ConfigKeys::ERRORS_SHOW_TRACE, false) : false;

        $genericUserMessage = 'An unexpected error occurred.';
        $userMessage = ($e instanceof HttpException) ? (string)$e->getUserMessage() : $genericUserMessage;

        if ($sanitizeUser) {
            $userMessage = HttpException::sanitizeMessage($userMessage);
        }

        // Determine "origin" (where the underlying error happened)
        $origin = $e;
        $wrappedByHttp = false;

        if ($e instanceof HttpException) {
            $prev = self::safePrevious($e);
            if ($prev) {
                $origin = $prev; // show exception/location/trace from the *cause*
                $wrappedByHttp = true;
            } else {
                $wrappedByHttp = true; // still wrapped, just no previous
            }
        }

        // JSON for action/ajax/SSE
        if ($req && self::isActionRequest($req)) {
            $details = null;

            if ($debug) {
                // Prefer HttpException debug details if provided
                if ($e instanceof HttpException) {
                    $provided = $e->getDebugDetails();
                    if ($provided !== null && $provided !== '') {
                        $details = (string)$provided;
                    }
                }

                // Fallback to origin exception debug text
                if ($details === null) {
                    $details = self::buildDebugText($origin, $showTrace);
                } elseif ($showTrace) {
                    $details .= "\n\n" . self::trimTrace(self::safeTrace($origin));
                }

                if ($sanitizeDebug && $details !== null) $details = HttpException::sanitizeMessage($details);
            }

            return new JsonResponse(
                array(
                    'error' => true,
                    'message' => $debug ? self::safeMessage($e) : $userMessage,
                    'code' => self::safeCode($e),
                    // Show the *cause* class when wrapped, else the exception class
                    'exception' => $debug ? self::safeClass($origin) : null,
                    // Indicate wrapping in debug
                    'wrapped_as' => ($debug && $wrappedByHttp) ? self::safeClass($e) . ' (HTTP ' . (int)$status . ')' : null,
                    'details' => ($debug ? $details : null),
                ),
                $status
            );
        }

        // HTML with sections
        $sections = array(
            'Message' => $userMessage,
        );

        if ($debug) {
            $sections['Exception Message'] = self::safeMessage($origin);
            // Exception section shows the *cause* (origin)
            $sections['Exception'] = self::safeClass($origin);

            $chain = self::resolutionChain($e);
            if ($chain) {
                if ($sanitizeDebug) $chain = HttpException::sanitizeMessage($chain);
                $sections['Resolution Chain'] = $chain;
            }

            if($e instanceof ContextException) {
                $context = $e->context();
                if (!empty($context)) {
                    $ctxLines = array();
                    foreach ($context as $k => $v) {
                        $line = (string)$k . ': ' . var_export($v, true);
                        if ($sanitizeDebug) $line = HttpException::sanitizeMessage($line);
                        $ctxLines[] = $line;
                    }
                    $sections['Context'] = implode("\n", $ctxLines);
                }
            }

            // Indicate it was wrapped as HttpException
            if ($wrappedByHttp) {
                $wrappedLine = self::safeClass($e) . ' (HTTP ' . (int)$status . ')';
                if ($sanitizeDebug) $wrappedLine = HttpException::sanitizeMessage($wrappedLine);
                $sections['Wrapped As'] = $wrappedLine;
            }

            // Location should reflect origin exception
            $location = self::safeFileLine($origin);
            if ($sanitizeDebug && $location !== null) $location = HttpException::sanitizeMessage($location);
            $sections['Location'] = $location;

            // If HttpException debug details provided, show them
            if ($e instanceof HttpException) {
                $provided = $e->getDebugDetails();
                if ($provided !== null && $provided !== '') {
                    $dbg = (string)$provided;
                    if ($sanitizeDebug) $dbg = HttpException::sanitizeMessage($dbg);
                    $sections['Details'] = $dbg;
                }
            }

            if ($showTrace) {
                /*$trace = self::trimTrace(self::safeTrace($origin));
                if ($sanitizeDebug) $trace = HttpException::sanitizeMessage($trace);
                $sections['Trace'] = $trace;*/
                $trace = self::compactTrace($origin);
                if ($sanitizeDebug) $trace = HttpException::sanitizeMessage($trace);
                $sections['Trace'] = $trace;
            }
        }

        return ErrorResponse::sections($status, 'Application Error', $sections);
    }

    // src/Http/ExceptionResponder.php  (add)
    private static function compactTrace($e, $max = 18)
    {
        if (!is_object($e) || !method_exists($e, 'getTrace')) return '';

        $out = array();
        $trace = $e->getTrace();

        foreach ($trace as $f) {
            // Drop the worst offenders
            if (isset($f['function']) && ($f['function'] === 'call_user_func' || $f['function'] === 'call_user_func_array')) {
                continue;
            }
            if (isset($f['class']) && $f['class'] === 'Closure') {
                continue;
            }
            if (isset($f['class']) && strpos($f['class'], 'boru\\dweb\\Kernel\\Container') === 0) {
                // container internals are rarely helpful once we have the resolution chain
                continue;
            }

            $call = '';
            if (isset($f['class'])) $call .= $f['class'] . '::';
            $call .= isset($f['function']) ? $f['function'] : '(unknown)';

            $loc = '';
            if (isset($f['file'])) $loc .= $f['file'];
            if (isset($f['line'])) $loc .= ':' . $f['line'];

            $out[] = $call . ($loc ? " @ " . $loc : '');

            if (count($out) >= $max) break;
        }

        return implode("\n", $out);
    }


    private static function trimTrace($trace)
    {
        $trace = (string)$trace;
        if ($trace === '') return '';

        $lines = preg_split("/\\r?\\n/", $trace);
        if (!$lines) return $trace;

        $dropPrefixes = array(
            'boru\\dweb\\Http\\ExceptionResponder',
            'boru\\dweb\\WebUI->render',
            'boru\\dweb\\WebUI->ensureDefaultsPublishedIfNeeded',
        );

        while (!empty($lines)) {
            $line = $lines[0];
            $matched = false;

            foreach ($dropPrefixes as $needle) {
                if (strpos($line, $needle) !== false) {
                    array_shift($lines);
                    $matched = true;
                    break;
                }
            }

            if (!$matched) break;
        }

        if (!empty($lines) && strpos($lines[0], 'HttpException::') !== false) {
            array_shift($lines);
        }

        return implode("\n", $lines);
    }

    private static function buildDebugText($e, $showTrace)
    {
        $out = self::safeMessage($e) . "\n\n" . self::safeFileLine($e);
        if ($showTrace) {
            $t = self::trimTrace(self::safeTrace($e));
            if ($t !== '') $out .= "\n\n" . $t;
        }
        return $out;
    }

    private static function safePrevious($e)
    {
        if (!is_object($e) || !method_exists($e, 'getPrevious')) return null;
        $p = $e->getPrevious();
        return ($p instanceof \Exception) ? $p : null;
    }

    private static function statusFromException($e)
    {
        if ($e instanceof HttpException) {
            $s = (int)$e->getStatusCode();
            if ($s >= 400 && $s <= 599) return $s;
        }

        if (is_object($e) && method_exists($e, 'getStatusCode')) {
            $s = (int)$e->getStatusCode();
            if ($s >= 400 && $s <= 599) return $s;
        }

        if ($e instanceof \InvalidArgumentException) {
            return 400;
        }

        $msg = strtolower(self::safeMessage($e));
        if (strpos($msg, 'publish') !== false || strpos($msg, 'initializ') !== false) {
            return 503;
        }

        return 500;
    }

    private static function isActionRequest(Request $req)
    {
        if ($req->param('action', null) !== null) return true;

        $accept = strtolower((string)$req->header('accept', ''));
        if (strpos($accept, 'application/json') !== false) return true;
        if (strpos($accept, 'text/event-stream') !== false) return true;

        $ct = strtolower((string)$req->header('content-type', ''));
        if (strpos($ct, 'application/json') !== false) return true;

        return false;
    }

    private static function safeMessage($e)
    {
        return (is_object($e) && method_exists($e, 'getMessage')) ? (string)$e->getMessage() : 'Error';
    }

    private static function safeFileLine($e)
    {
        if (!is_object($e)) return '';
        if (!method_exists($e, 'getFile') || !method_exists($e, 'getLine')) return '';
        return (string)$e->getFile() . ':' . (string)$e->getLine();
    }

    private static function safeTrace($e)
    {
        return (is_object($e) && method_exists($e, 'getTraceAsString')) ? (string)$e->getTraceAsString() : '';
    }

    private static function safeCode($e)
    {
        return (is_object($e) && method_exists($e, 'getCode')) ? $e->getCode() : 0;
    }

    private static function safeClass($e)
    {
        return is_object($e) ? get_class($e) : 'unknown';
    }

    private static function resolutionChain($e)
    {
        $chain = array();

        while ($e) {
            if ($e instanceof \boru\dweb\Exceptions\ServiceBuildException) {
                $label = $e->serviceLabel();
                $chain[] = $label ? $label : $e->serviceId();
            }
            $e = method_exists($e, 'getPrevious') ? $e->getPrevious() : null;
        }

        if (empty($chain)) return null;

        // Outer -> inner reads best
        return implode("  \n→ ", $chain);
    }
}
