<?php
namespace boru\dweb\Kernel;

use boru\dweb\Exceptions\ServiceBuildException;

/**
 * Tiny explicit container with callable factory support.
 *
 * Supports:
 * - set('id', $value)                    // instance value
 * - set('id', function(Container $c){})  // shared factory (cached)
 * - factory('id', function(Container $c){}) // non-shared factory (no cache)
 * - get('id')
 */
class Container
{
    /** @var array<string, mixed> */
    private $entries = array();

    /** @var array<string, bool> */
    private $isFactory = array();

    /** @var array<string, bool> */
    private $isSharedFactory = array();

    /** @var array<string, mixed> */
    private $resolved = array();

    /** @var array<string,string> */
    private $labels = array(); // NEW

    public function label($id, $label)
    {
        $this->labels[(string)$id] = (string)$label;
    }

    private function labelFor($id)
    {
        $id = (string)$id;
        return isset($this->labels[$id]) ? $this->labels[$id] : null;
    }

    /**
     * Register an entry (value OR shared factory).
     *
     * @param string $id
     * @param mixed $valueOrFactory
     * @return $this
     */
    public function set($id, $valueOrFactory)
    {
        $id = (string)$id;

        $this->entries[$id] = $valueOrFactory;

        $isCallable = is_callable($valueOrFactory);
        $this->isFactory[$id] = $isCallable;
        $this->isSharedFactory[$id] = $isCallable ? true : false;

        unset($this->resolved[$id]);
        return $this;
    }

    /**
     * Register a NON-shared factory (new instance each time).
     *
     * @param string $id
     * @param callable $factory function(Container $c)
     * @return $this
     */
    public function factory($id, $factory)
    {
        $id = (string)$id;

        if (!is_callable($factory)) {
            throw new \InvalidArgumentException("Container::factory expects a callable for: " . $id);
        }

        $this->entries[$id] = $factory;
        $this->isFactory[$id] = true;
        $this->isSharedFactory[$id] = false;

        unset($this->resolved[$id]);
        return $this;
    }

    /**
     * @param string $id
     * @return bool
     */
    public function has($id)
    {
        $id = (string)$id;
        return array_key_exists($id, $this->entries);
    }

    /**
     * @param string $id
     * @return mixed
     * @throws \RuntimeException
     */
    public function get($id)
    {
        $id = (string)$id;

        if (!array_key_exists($id, $this->entries)) {
            throw new \boru\dweb\Exceptions\ContainerEntryNotFound($id, array_keys($this->entries));
        }

        // Return cached shared factory result
        if (isset($this->resolved[$id])) {
            return $this->resolved[$id];
        }

        $entry = $this->entries[$id];

        // Value entry
        if (empty($this->isFactory[$id])) {
            return $entry;
        }

        // Factory entry
        try {
            $value = call_user_func($entry, $this);
        } catch (\Exception $e) {
            throw new ServiceBuildException($id, $this->labelFor($id), $e);
        }

        // Cache if shared
        if (!empty($this->isSharedFactory[$id])) {
            $this->resolved[$id] = $value;
        }

        return $value;
    }
}
