<?php
namespace boru\dweb\Kernel;

use boru\dweb\Contracts\ModuleInterface;
use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Contracts\RendererInterface;
use boru\dweb\Routing\CanonicalUrl;
use boru\dweb\Routing\RouteCollection;
use boru\dweb\Mvc\DefaultViewFactory;
use boru\dweb\Mvc\DefaultActionFactory;
use boru\dweb\Mvc\ModuleNamespaceResolver;

class ModuleManager implements \IteratorAggregate
{
    /** @var ModuleInterface[] moduleName => module */
    private $modules = array();

    /** @var DefaultViewFactory[] moduleName => factory */
    private $viewFactories = array();

    /** @var DefaultActionFactory[] moduleName => factory */
    private $actionFactories = array();

    /**
     * Allow `foreach ($moduleManager as $m)` everywhere (extensions, publishers, etc).
     * @return \Traversable
     */
    public function getIterator()
    {
        return new \ArrayIterator($this->modules);
    }
    
    public function add(ModuleInterface $module)
    {
        $this->modules[$module->getName()] = $module;
        return $this;
    }

    /** @return ModuleInterface[] */
    public function all()
    {
        return array_values($this->modules);
    }

    /** @return ModuleInterface|null */
    public function get($moduleName)
    {
        $moduleName = (string)$moduleName;
        return isset($this->modules[$moduleName]) ? $this->modules[$moduleName] : null;
    }

    /**
     * Boot modules and build a RouteCollection.
     *
     * @param Container $container
     * @param SettingsInterface $settings
     * @return RouteCollection
     */
    public function boot(Container $container, SettingsInterface $settings)
    {
        $routes = new RouteCollection();

        foreach ($this->modules as $m) {
            $m->register($container, $settings);

            $moduleRoutes = $routes->forModule($m->getName());
            $m->routes($moduleRoutes, $container, $settings);
        }

        return $routes;
    }

    /**
     * @param string $moduleName
     * @param Container $container
     * @return DefaultViewFactory
     */
    public function getViewFactory($moduleName, Container $container)
    {
        $moduleName = (string)$moduleName;
        if (isset($this->viewFactories[$moduleName])) return $this->viewFactories[$moduleName];

        $module = $this->get($moduleName);
        if (!$module) throw new \RuntimeException('Unknown module: ' . $moduleName);

        $renderer  = $container->get(RendererInterface::class);
        $canonical = $container->get(CanonicalUrl::class);
        $resolver  = $this->getNamespaceResolver($container);

        $ctx = $this->buildControllerContext($module, $container);
        $factory = new DefaultViewFactory($renderer, $canonical, $module, $resolver, $ctx);
        $this->viewFactories[$moduleName] = $factory;
        return $factory;
    }

    /**
     * @param string $moduleName
     * @param Container $container
     * @return DefaultActionFactory
     */
    public function getActionFactory($moduleName, Container $container)
    {
        $moduleName = (string)$moduleName;
        if (isset($this->actionFactories[$moduleName])) return $this->actionFactories[$moduleName];

        $module = $this->get($moduleName);
        if (!$module) throw new \RuntimeException('Unknown module: ' . $moduleName);

        $renderer  = $container->get(RendererInterface::class);
        $canonical = $container->get(CanonicalUrl::class);
        $resolver  = $this->getNamespaceResolver($container);

        $ctx = $this->buildControllerContext($module, $container);

        $factory = new DefaultActionFactory($renderer, $canonical, $module, $resolver, $ctx);
        $this->actionFactories[$moduleName] = $factory;
        return $factory;
    }

    private function getNamespaceResolver(Container $c)
    {
        // Prefer container-provided resolver if host overrides, otherwise default
        if ($c->has(ModuleNamespaceResolver::class)) {
            return $c->get(ModuleNamespaceResolver::class);
        }
        return new ModuleNamespaceResolver();
    }

    private function buildControllerContext(ModuleInterface $module, Container $container)
    {
        $settings = $container->get(\boru\dweb\Contracts\SettingsInterface::class);
        $canonical = $container->get(\boru\dweb\Routing\CanonicalUrl::class);

        $dweb = new \boru\dweb\Mvc\DwebHelper($canonical, $module->getName());

        $ctx = new \boru\dweb\Support\ControllerContext($settings, $dweb);

        // Optional services if present
        if ($container->has(\boru\dweb\Contracts\LoggerInterface::class)) {
            $ctx->withLogger($container->get(\boru\dweb\Contracts\LoggerInterface::class));
        }
        if ($container->has(\boru\dweb\Contracts\CliInvokerInterface::class)) {
            $ctx->withCli($container->get(\boru\dweb\Contracts\CliInvokerInterface::class));
        }
        if ($container->has(\boru\dweb\Assets\AssetManager::class)) {
            $ctx->withAssets($container->get(\boru\dweb\Assets\AssetManager::class));
        }
        // If you have an AssetManagerInterface, prefer that instead.

        return $ctx;
    }
}
