# Rendering in boru/dweb

Rendering in `boru/dweb` is **explicit, layered, and contract-driven**.

There is no implicit controller rendering, no auto-mapped templates,
and no framework-owned UI.

---

## Core Principles

- Controllers explicitly return responses
- Rendering is not tied to routing
- Templates are module-scoped
- Layouts are explicit
- Fragments are first-class
- Rendering engines are swappable

---

## Controllers: Views and Actions

Both **Views** and **Actions**:

- may render templates
- may return fragments
- may return JSON
- may return SSE streams

The distinction is **semantic**, not technical.

| Type | Typical use |
|----|----|
| View | Pages, fragments, layout composition |
| Action | APIs, form handlers, SSE streams |

---

## Rendering Flow

```

Controller
→ Renderer
→ TemplateLocator
→ Template
→ Layout (optional)
→ Response

```

Nothing is inferred automatically.

---

## Rendering a Full Page

```php
return $this->render(
    'layouts/default',
    'home/index',
    ['title' => 'Hello']
);
```

---

## Fragments (Layout-Free Rendering)

Fragments are templates rendered **without layouts**.

```php
return $this->fragment('partials/item', ['item' => $item]);
```

Fragments are ideal for:

* HTMX
* AJAX
* SSE
* progressive enhancement

---

## Sections and Layout Composition

Controllers may populate named sections explicitly.

```php
$this->sectionSet('sidebar',
    $this->renderPartialToString('partials/sidebar')
);
```

Layouts render sections explicitly:

```smarty
<aside id="{$dweb->sectionId('sidebar')}">
  {$sections.sidebar|default:'' nofilter}
</aside>

<main id="{$dweb->sectionId('content')}">
  {$sections.content|default:$content nofilter}
</main>
```

There is:

* no output buffering
* no implicit capture
* no magic slots

---

### Section templates: `sectionTemplate()`

For the common case “render a template into a named section”, views may use
the convenience helper `sectionTemplate()`:

```php
// In a View extending AbstractView

public function __invoke()
{
    // Main page content
    $this->sectionTemplate(
        'content',
        'home/index',
        [
            'title'   => 'Hello',
            'somevar' => 'value',
        ]
    );

    // Sidebar section (e.g. templates/sections/sidebar.tpl or partials/sidebar.tpl,
    // depending on your convention in sectionTemplate())
    $this->sectionTemplate(
        'sidebar',
        'partials/sidebar',
        [
            'items' => $items,
        ]
    );

    return $this->layoutWithSections('layouts/default');
}
```

Under the hood:

- `sectionTemplate($name, $templateId, $data)` calls
  `renderPartialToString($templateId, $data)` and then `sectionSet($name, $html)`.
- Sections are stored in a `SectionBag` on the view instance.

You can still use `sectionSet()` / `sectionAppend()` directly if you want to
build sections from arbitrary HTML or compose them manually.

---

### Rendering a layout with sections: `layoutWithSections()`

When you want full layout resolution (global defaults, cross-module overrides)
**and** explicit sections, use `layoutWithSections()`:

```php
public function __invoke()
{
    $this->sectionTemplate(
        'content',
        'home/index',
        ['title' => 'Hello']
    );

    $this->sectionTemplate(
        'sidebar',
        'partials/sidebar',
        ['items' => $items]
    );

    return $this->layoutWithSections('layouts/default');
}
```

Behavior:

- If no `content` section has been set, `layoutWithSections()` ensures an empty
  `content` value exists.
- It builds `$layoutData`:

  ```php
  $layoutData = [
      'content'  => $sections->get('content'),
      'sections' => $sections->all(),
  ];
  ```

- It delegates to the renderer via:

  ```php
  $html = $this->renderer->renderLayoutWithSections(
      $this->moduleName,
      $layoutId,
      $layoutData
  );
  ```

- The renderer (e.g. `SmartyRenderer`) is responsible for:
  - applying layout resolution (global default layouts, module overrides),
  - rendering the actual layout template with `$content` and `$sections` available.

In layouts, you continue to access sections the same way:

```smarty
<header id="{$dweb->sectionId('header')}">
  {$sections.header|default:'' nofilter}
</header>

<aside id="{$dweb->sectionId('sidebar')}">
  {$sections.sidebar|default:'' nofilter}
</aside>

<main id="{$dweb->sectionId('content')}">
  {$sections.content|default:$content nofilter}
</main>
```

---

## Fragment → Section Targeting

Fragments may declare which section they update:

```php
return $this->fragmentToSection(
    'sidebar',
    'partials/sidebar',
    ['items' => $items]
);
```

Response headers indicate intent:

```
X-DWeb-Fragment: Skeleton.sidebar
X-DWeb-Section: sidebar
```

---

## Server-Sent Events (SSE)

dweb supports **native SSE streaming** via `SseResponse`.

SSE responses are first-class and integrate with the standard response emitter.

### Streaming SSE from an Action

```php
return SseResponse::stream(function ($send) {
    $send('<div>Starting...</div>', 'status');

    for ($i = 1; $i <= 5; $i++) {
        $send('<span>' . $i . '</span>', 'content');
        sleep(1);
    }

    $send('<div>Done</div>', 'status');
});
```

SSE is ideal for:

* live updates
* progress indicators
* streaming AI output
* HTMX SSE integration

---

## HTMX + SSE

HTMX can consume SSE directly using its `sse` extension.

dweb SSE streams typically emit **HTML fragments**, not JSON,
allowing server-rendered UI updates with zero client JS.

---

## Renderer Implementations

The framework does not mandate a renderer.

Current implementation:

* Smarty (via shim)

Possible future renderers:

* Native PHP
* Twig
* Static pre-rendering

Controllers do not change when renderers change.

Views and Actions access URL helpers via `$this->dweb()`, provided by `ControllerContext`.

---

## Design Rules

* Controllers control responses
* Renderers control rendering
* Templates contain presentation only
* No auto-mapping of templates
* No hidden behavior

If something renders, it should be obvious where it came from.