# ControllerContext

`ControllerContext` is a **typed façade** that provides controllers (Views and Actions)
with access to framework services **without exposing the DI container**.

It exists to balance:
- explicit dependencies
- controller ergonomics
- architectural boundaries

---

## Why ControllerContext Exists

Direct container access inside controllers was intentionally avoided to prevent:

- service-locator anti-patterns
- hidden dependencies
- hard-to-test controllers
- tight coupling to framework internals

However, real applications require access to shared services such as:

- URL generation
- assets
- CLI tooling
- logging
- configuration

`ControllerContext` provides these capabilities **explicitly and safely**.

---

## What Controllers Receive

Every controller that extends `AbstractView` or `AbstractAction`
receives a `ControllerContext` automatically.

Controllers do **not** receive the container.

---

## Accessing the Context

Inside a controller:

```php
$this->ctx()
````

Convenience helpers are provided:

```php
$this->dweb();     // URL + routing helpers
$this->cli();      // CLI invoker (optional)
$this->assets();   // Asset manager (optional)
$this->settings(); // Read-only configuration
$this->logger();   // Logger (optional)
```

If a service is not available, accessing it will throw a clear exception.

---

## dweb() — URL & Routing Helper

```php
$this->dweb()->view('home');
$this->dweb()->view('results', $id);
$this->dweb()->action('ping');
```

This is the same helper exposed to templates as `$dweb`.

---

## cli() — Running dweb CLI Commands

Controllers may invoke **registered dweb CLI commands**:

```php
$result = $this->cli()->run('publish', [], [
    'path' => '/tmp/assets'
]);

if (!$result->ok()) {
    throw new \RuntimeException($result->stderr);
}
```

Notes:

* Only registered commands may be invoked
* Commands run via `vendor/bin/dweb`
* Results include exit code, stdout, and stderr
* CLI invocation is optional and config-controlled

---

## assets() — Asset Registration

If an AssetManager is present, controllers may register assets:

```php
$this->assets()->css('Skeleton/css/demo.css');
$this->assets()->js('Skeleton/js/demo.js');
```

Assets are section-aware and renderer-agnostic.

---

## settings() — Read-Only Configuration

Controllers may read configuration values:

```php
$debug = $this->settings()->get(ConfigKeys::DEBUG_ENABLED);
```

Controllers must **not** mutate configuration.

---

## logger() — Logging

If a logger is available:

```php
$this->logger()->info('Page loaded', [
    'module' => 'Skeleton'
]);
```

Logging is optional and environment-dependent.

---

## How ControllerContext Is Built

`ControllerContext` is constructed by the framework during controller creation.

It is assembled from:

* SettingsInterface
* CanonicalUrl
* module name
* optional services (CLI, assets, logger)

This occurs in the controller factory layer, **not** in controllers.

---

## Design Rules

* Controllers never access the container
* Context is injected automatically
* Services are explicit and typed
* No global state
* No magic resolution

If a controller needs a new capability, it should be added
to `ControllerContext`, not fetched from the container.

---

## Summary

`ControllerContext` provides:

* clean controller APIs
* strong architectural boundaries
* extensibility without chaos
* a stable foundation for future features

Controllers stay simple.
Framework complexity stays contained.