# Realtime & Socket.IO Integration

dweb includes a **minimal, explicit realtime layer** based on a separate
Node.js Socket.IO server.

The goal is to provide:

- a stable **config surface** in PHP
- a reusable **token service** for auth
- a small **PHP → Node publish bridge**
- a tiny **JS client helper** (`dweb.socket.js`)

while keeping application semantics (events, channels, UI) inside your
**host app modules**, not the framework core.

---

## 1. Architecture

### 1.1 Components

**On the PHP side:**

- `ConfigKeys`:
  - `dweb.socket.enabled`
  - `dweb.socket.auth.mode` (`jwt` or `opaque`)
  - `dweb.socket.jwt.secret`
  - `dweb.socket.token.ttl`
  - `dweb.socket.api.secret`
  - `dweb.socket.public_url`
  - `dweb.socket.node.bin`
  - `dweb.socket.node.script`
  - `dweb.socket.node.port`

- Services:
  - `SocketTokenService`
    - Issues JWT or opaque tokens for a subject + claims.
  - `SocketPublisher`
    - Sends publish requests from PHP to the Node server (`/publish` API).

- Core module:
  - `boru\dweb\Modules\Core\CoreModule`
    - Route: `core.socketToken`
      - Generic infrastructure endpoint for issuing socket tokens.

- CLI:
  - `socket:serve` (`SocketServeCommand`)
    - Starts the Node Socket.IO server using config.
    - Exports relevant env vars for Node:
      - `DWEB_SOCKET_AUTH_MODE`
      - `DWEB_SOCKET_JWT_SECRET`
      - `DWEB_SOCKET_API_SECRET`

**On the Node side:**

- `node/dweb-socket-server.js`:
  - A small Socket.IO server process.
  - Authenticates incoming connections using the token:
    - `jwt` mode: HS256 JWT via `jsonwebtoken`.
    - `opaque` mode: currently a stub (log + accept; validation TBD).
  - Joins per-user rooms:
    - If JWT payload has `sub`, the socket joins room `user:{sub}`.
  - Provides a simple demo `ping`/`pong` channel.
  - Exposes a `POST /publish` HTTP endpoint for PHP:
    - Body: `{"channel": "room-or-null", "event": "name", "payload": {...}}`
    - Header: `X-Dweb-Socket-Api-Secret: <secret>`

**On the JS side:**

- `dweb.socket.js`:
  - Reads config from `window.DWEB_SOCKET_CONFIG`.
  - Fetches a token from a configurable endpoint (default `core.socketToken`).
  - Establishes a shared Socket.IO connection using `io(...)`.
  - Exposes a small, framework-level JS API:
    - `dweb.socket.ensureConnected(cb)`
    - `dweb.socket.on(event, handler)`
    - `dweb.socket.emit(event, payload)`
    - `dweb.socket.disconnect()`

---

## 2. PHP Configuration

All socket settings live in `DwebConfig` / `ConfigKeys`:

```php
$config->set(\boru\dweb\Config\ConfigKeys::SOCKET_ENABLED, true);
$config->set(\boru\dweb\Config\ConfigKeys::SOCKET_AUTH_MODE, 'jwt');
$config->set(\boru\dweb\Config\ConfigKeys::SOCKET_JWT_SECRET, 'your-hs256-secret');
$config->set(\boru\dweb\Config\ConfigKeys::SOCKET_TOKEN_TTL, 3600);

// Optional: where clients connect
$config->set(\boru\dweb\Config\ConfigKeys::SOCKET_PUBLIC_URL, 'http://localhost:3001');

// Optional: Node process settings (socket:serve uses these)
$config->set(\boru\dweb\Config\ConfigKeys::SOCKET_NODE_BIN, 'node');
$config->set(\boru\dweb\Config\ConfigKeys::SOCKET_NODE_SCRIPT, null); // default: node/dweb-socket-server.js
$config->set(\boru\dweb\Config\ConfigKeys::SOCKET_NODE_PORT, 3001);

// PHP -> Node publish API secret (used by SocketPublisher and Node /publish)
$config->set(\boru\dweb\Config\ConfigKeys::SOCKET_API_SECRET, 'some-long-random-string');
```

Make sure `CoreModule` is added to your modules list in the env/bootstrap:

```php
use boru\dweb\Modules\Core\CoreModule;
use boru\dweb\Modules\Skeleton\SkeletonModule;

$configureModules = function (\boru\dweb\Kernel\ModuleManager $mm, $c, $cfg) {
    $mm->add(new CoreModule());
    $mm->add(new SkeletonModule());
    // ... your other modules ...
};
```

This enables the `core.socketToken` action used by the JS helper.

---

## 3. Running the Socket Server

Node dependencies (from the framework root):

```bash
npm init -y
npm install socket.io jsonwebtoken
```

Then run the server either manually:

```bash
node node/dweb-socket-server.js --port=3001
```

or via the dweb CLI:

```bash
vendor/bin/dweb socket:serve \
  --env-bootstrap=/abs/path/to/env.php \
  --port=3001
```

`socket:serve` will:

- Use your `DwebConfig` to determine:
  - auth mode
  - JWT secret
  - Node bin, script, and port
  - API secret (`SOCKET_API_SECRET`)
- Export corresponding env vars for the Node process.

---

## 4. Frontend Wiring

### 4.1 Core socket config fragment

Core provides a small template fragment that emits
`window.DWEB_SOCKET_CONFIG` for the current request.

In Smarty:

```smarty
{* Socket.io JS *}
<script src="https://cdn.socket.io/4.7.2/socket.io.min.js"></script>

{* DWeb Socket Config *}
{include file=$dweb->path('core/socket_config','core')}

{* DWeb Socket JS *}
<script src="{$dweb->assetUrl('dweb/dweb.socket.js')|escape}"></script>
```

This should usually live in your **base layout**, so any view can opt into
realtime by simply calling `dweb.socket.*` in its JS.

### 4.2 `window.DWEB_SOCKET_CONFIG`

`socket_config.tpl` sets up a basic config object:

```js
window.DWEB_SOCKET_CONFIG = {
    enabled: true,                     // from dweb.socket.enabled
    socketUrl: "http://localhost:3001",// from dweb.socket.public_url (or null)
    tokenUrl: null,                    // default: ?action=core.socketToken
    debug: false
};
```

Host apps may override:

- `socketUrl` — if the socket server is on a different host/port.
- `tokenUrl` — if you want to use a custom token endpoint instead of `core.socketToken`.
- `debug` — set to `true` to log extra messages.

---

## 5. JS API: `dweb.socket.*`

After including Socket.IO and `dweb.socket.js`, the framework exposes:

```js
dweb.socket.ensureConnected(cb)
dweb.socket.on(event, handler)
dweb.socket.emit(event, payload)
dweb.socket.disconnect()
```

### 5.1 `dweb.socket.ensureConnected(cb)`

Ensures there is an active connection, then calls:

```js
cb(err, socket)
```

- `err` — `Error` or `null`.
- `socket` — the underlying Socket.IO client instance returned by `io()`.

The connection is:

- Established lazily (on first use).
- Shared across all callers for the duration of the page.
- Authenticated via a token fetched from `DWEB_SOCKET_CONFIG.tokenUrl` (or the default core endpoint).

Example:

```js
dweb.socket.ensureConnected(function (err, socket) {
    if (err) {
        console.error('Failed to connect:', err);
        return;
    }

    console.log('Connected, id=', socket.id);

    // You can still use raw Socket.IO APIs if you wish:
    socket.emit('ping', { foo: 'bar' });
});
```

### 5.2 `dweb.socket.on(event, handler)`

Registers an event handler. This is the **preferred** way to subscribe to
events (including `connect` / `disconnect`).

Key behavior:

- Handlers are stored in an internal registry.
- When the socket connects or reconnects, all registered handlers are
  wired to the underlying Socket.IO instance.
- Duplicate registrations of the **same function** for the same event
  are ignored.

Special events:

- `'connect'` — fired by dweb.socket when Socket.IO’s `connect` event
  occurs. The handler receives the raw `socket` instance:

  ```js
  dweb.socket.on('connect', function (socket) {
      console.log('Connected via dweb.socket, id=', socket.id);
  });
  ```

- `'disconnect'` — fired when the underlying socket disconnects. The
  handler receives the reason string:

  ```js
  dweb.socket.on('disconnect', function (reason) {
      console.log('Disconnected:', reason);
  });
  ```

For all other events, the handler receives the event payload:

```js
dweb.socket.on('notification', function (payload) {
    console.log('Notification received:', payload);
});
```

### 5.3 `dweb.socket.emit(event, payload)`

Emits an event with a payload via the shared connection.

If the connection is not yet established, it will be created first.

```js
dweb.socket.emit('ping', { at: new Date().toISOString() });
```

Errors (e.g., failure to connect) are logged to `console.error` when
`debug` is enabled or when a connection cannot be created.

### 5.4 `dweb.socket.disconnect()`

Disconnects the shared socket (if any) and resets the internal connection
state for this page load.

Registered handlers are kept in memory so that a subsequent call to
`ensureConnected()` will:

- Establish a new connection.
- Re‑attach all previously registered handlers.

Example:

```js
// User opts out of realtime:
dweb.socket.disconnect();
```

---

## 6. Server-Side Publishing from Modules

Any module can push messages to connected clients via `SocketPublisher`,
which is registered in the container and attached to controller context.

From a controller (View or Action):

```php
$publisher = $this->ctx()->socketPublisher();

if ($publisher) {
    // Broadcast to all connected clients
    $publisher->publish(
        null, // null channel => broadcast
        'realtimeDemo',
        array('message' => 'Hello everyone!', 'at' => date('c'))
    );

    // Target a specific user (JWT sub == userId)
    $userId = 'demo-user';
    $publisher->publish(
        'user:' . $userId,
        'notification',
        array('message' => 'Hi ' . $userId, 'at' => date('c'))
    );
}
```

On the Node side, when a client connects in `jwt` mode, the server:

- Verifies the JWT.
- Extracts `sub` from the payload.
- Joins room `user:{sub}` for targeted notifications.

On the JS side, modules only need to subscribe:

```js
dweb.socket.on('realtimeDemo', function (payload) {
    console.log('Broadcast event:', payload);
});

dweb.socket.on('notification', function (payload) {
    console.log('User notification:', payload);
});
```

---

## 7. Overriding the Token Endpoint

By default, `dweb.socket.js` calls:

```text
?action=core.socketToken
```

to fetch a token. This endpoint is provided by the `CoreModule` and uses
`SocketTokenService` with a generic notion of subject.

Host apps that want full control can:

1. Implement their own Action (e.g. `Auth.socketToken`), which:
   - Looks up the current authenticated user.
   - Issues a token with `SocketTokenService` (and custom claims).
2. Set a custom token URL in `DWEB_SOCKET_CONFIG`:

   ```smarty
   <script>
   window.DWEB_SOCKET_CONFIG = {
       enabled: true,
       socketUrl: "https://socket.example.com",
       tokenUrl: "/api/Auth/socketToken",
       debug: false
   };
   </script>
   ```

In this mode, `core.socketToken` can be left enabled or disabled; it is
no longer used by the frontend.

---

This realtime/socket layer is intentionally small and explicit. It
handles the plumbing—config, token issuance, connection, and publish
bridge—so your modules and host app can focus on:

- which events exist,
- when to emit them,
- and how the UI should react.
```