#!/usr/bin/env node

/**
 * dweb Socket.IO server
 *
 * Responsibilities:
 * - Start a Socket.IO server on a given port.
 * - Read auth mode + secrets from environment:
 *     DWEB_SOCKET_AUTH_MODE = 'jwt' | 'opaque'
 *     DWEB_SOCKET_JWT_SECRET = shared HS256 secret (required in jwt mode)
 * - On connection:
 *     - Extract token from:
 *         socket.handshake.auth.token
 *         OR socket.handshake.query.token
 *     - If auth mode = jwt:
 *         - Verify token with jsonwebtoken + secret
 *         - Attach decoded payload to socket.data.user
 *     - If auth mode = opaque:
 *         - Currently: log a warning and accept (stub – real validation TBD).
 * - Demo channel:
 *     client: socket.emit('ping', { foo: 'bar' })
 *     server: socket.emit('pong', { now, echo: payload })
 */

const http = require('http');
const { Server } = require('socket.io');
const jwt = require('jsonwebtoken');
const url = require('url'); // <-- add this


// ---------- CLI args: --port=NNNN ----------

let port = 3001;
process.argv.slice(2).forEach(arg => {
    if (arg.indexOf('--port=') === 0) {
        const v = parseInt(arg.split('=')[1], 10);
        if (!isNaN(v) && v > 0) {
            port = v;
        }
    }
});

// ---------- Environment config ----------

const AUTH_MODE = (process.env.DWEB_SOCKET_AUTH_MODE || 'jwt').toLowerCase();
const JWT_SECRET = process.env.DWEB_SOCKET_JWT_SECRET || '';
const API_SECRET = process.env.DWEB_SOCKET_API_SECRET || '';

if (AUTH_MODE === 'jwt' && !JWT_SECRET) {
    console.error('[dweb-socket] ERROR: AUTH_MODE=jwt but DWEB_SOCKET_JWT_SECRET is not set.');
    process.exit(1);
}

console.log('[dweb-socket] Starting server...');
console.log('[dweb-socket]  port     :', port);
console.log('[dweb-socket]  authMode :', AUTH_MODE);

// ---------- HTTP + Socket.IO setup ----------

const httpServer = http.createServer((req, res) => {
    const parsed = url.parse(req.url, true);

    // Simple publish API for PHP backend:
    //   POST /publish
    //   Headers:
    //     X-Dweb-Socket-Api-Secret: <API_SECRET>
    //   Body (JSON):
    //     { "channel": "room-or-null", "event": "eventName", "payload": { ... } }
    if ((req.method === 'POST' || req.method === 'GET') && parsed.pathname === '/publish') {
        if (!API_SECRET) {
            res.statusCode = 500;
            res.end('API_SECRET not configured');
            return;
        }

        const headerSecret = req.headers['x-dweb-socket-api-secret'];
        if (headerSecret !== API_SECRET) {
            res.statusCode = 403;
            res.end('Forbidden');
            return;
        }

        let body = '';
        req.on('data', chunk => { body += chunk; });
        req.on('end', () => {
            try {
                const data = JSON.parse(body || '{}');
                const channel = data.channel || null;
                const event = data.event || 'message';
                const payload = data.payload || {};

                if (channel) {
                    io.to(channel).emit(event, payload);
                    console.log('[dweb-socket] publish -> channel=', channel, 'event=', event);
                } else {
                    io.emit(event, payload);
                    console.log('[dweb-socket] publish -> broadcast event=', event);
                }

                res.statusCode = 200;
                res.end('ok');
            } catch (e) {
                console.error('[dweb-socket] publish error:', e && e.message);
                res.statusCode = 400;
                res.end('bad json');
            }
        });
        return;
    }

    res.statusCode = 404;
    res.end('not found');
});

const io = new Server(httpServer, {
    cors: {
        origin: '*',
        methods: ['GET', 'POST'],
    }
});


// ---------- Auth helper ----------

function extractToken(socket) {
    // Preferred: socket.handshake.auth.token
    if (socket.handshake && socket.handshake.auth && socket.handshake.auth.token) {
        return socket.handshake.auth.token;
    }

    // Fallback: query token
    if (socket.handshake && socket.handshake.query && socket.handshake.query.token) {
        return socket.handshake.query.token;
    }

    return null;
}

function verifyJwtToken(token) {
    try {
        const decoded = jwt.verify(token, JWT_SECRET);
        return { ok: true, payload: decoded };
    } catch (err) {
        return { ok: false, error: err };
    }
}

// ---------- Connection handling ----------

io.on('connection', (socket) => {
    const token = extractToken(socket);

    if (!token) {
        console.warn('[dweb-socket] Connection missing token; disconnecting.');
        socket.emit('auth_error', { message: 'Missing token' });
        socket.disconnect(true);
        return;
    }

    if (AUTH_MODE === 'jwt') {
        const result = verifyJwtToken(token);
        if (!result.ok) {
            console.warn('[dweb-socket] Invalid JWT token:', result.error && result.error.message);
            socket.emit('auth_error', { message: 'Invalid token' });
            socket.disconnect(true);
            return;
        }

        // Attach user payload to socket for downstream handlers
        socket.data = socket.data || {};
        socket.data.user = result.payload;

        console.log('[dweb-socket] Client connected (jwt): sub =', result.payload && result.payload.sub);

        if (result.payload && result.payload.sub) {
            const room = 'user:' + result.payload.sub;
            socket.join(room);
            console.log('[dweb-socket] Joined room:', room);
        }
    } else if (AUTH_MODE === 'opaque') {
        // Stub: trust opaque tokens for now; real validation should call back into PHP
        console.warn('[dweb-socket] AUTH_MODE=opaque: token accepted without validation (stub).');
        socket.data = socket.data || {};
        socket.data.token = token;
    } else {
        console.warn('[dweb-socket] Unknown AUTH_MODE:', AUTH_MODE);
        socket.emit('auth_error', { message: 'Server misconfigured (unknown auth mode)' });
        socket.disconnect(true);
        return;
    }

    // Demo ping/pong channel
    socket.on('ping', (payload) => {
        const now = new Date().toISOString();
        socket.emit('pong', {
            now: now,
            echo: payload || null,
            user: socket.data && socket.data.user ? socket.data.user : null
        });
    });

    socket.on('join', (payload) => {
        const channel = payload && payload.channel;
        if (typeof channel !== 'string' || !channel) {
            console.warn('[dweb-socket] join: invalid channel', payload);
            return;
        }
        socket.join(channel);
        console.log('[dweb-socket] Socket joined channel:', channel);
    });


    socket.on('disconnect', (reason) => {
        console.log('[dweb-socket] Client disconnected:', reason);
    });
});

// ---------- Start server ----------

httpServer.listen(port, () => {
    console.log(`[dweb-socket] Listening on port ${port}`);
});
