// dweb.socket.js
// Lightweight Socket.IO helper for dweb.
//
// Responsibilities:
// - Read config from window.DWEB_SOCKET_CONFIG (if present).
// - Fetch a socket token from a configured endpoint (default: ?action=core.socketToken).
// - Establish a shared Socket.IO connection with auth {token}.
// - Provide a tiny API:
//     dweb.socket.ensureConnected(cb)
//     dweb.socket.on(event, handler)
//     dweb.socket.emit(event, payload)
//     dweb.socket.disconnect()
(function (global) {
    'use strict';

    var dweb = global.dweb = global.dweb || {};
    var cfg = global.DWEB_SOCKET_CONFIG || {};

    function pick(obj, key, fallback) {
        if (!obj || typeof obj !== 'object') return fallback;
        if (Object.prototype.hasOwnProperty.call(obj, key)) return obj[key];
        return fallback;
    }

    var socketUrl = pick(cfg, 'socketUrl', null);
    var tokenUrl  = pick(cfg, 'tokenUrl', null);
    var enabled   = pick(cfg, 'enabled', true);

    function resolveSocketUrl() {
        if (socketUrl && socketUrl !== '') return socketUrl;

        var loc = global.location || { protocol: 'http:', hostname: '127.0.0.1' };
        var host = loc.hostname || '127.0.0.1';

        return 'http://' + host + ':3001';
    }

    function resolveTokenUrl() {
        if (tokenUrl && tokenUrl !== '') return tokenUrl;
        throw Error('No tokenUrl configured for dweb.socket.');
    }

    function log() {
        if (cfg && cfg.debug) {
            if (global.console && console.log) {
                console.log.apply(console, ['[dweb.socket]'].concat(Array.prototype.slice.call(arguments)));
            }
        }
    }

    function errorLog() {
        if (global.console && console.error) {
            console.error.apply(console, ['[dweb.socket]'].concat(Array.prototype.slice.call(arguments)));
        }
    }

    var _socket = null;
    var _connecting = false;
    var _connectQueue = [];
    var _token = null;

    // Registry: eventName => [handler, ...]
    var _handlers = {};

    function httpGetJson(url, cb) {
        var xhr = new XMLHttpRequest();
        xhr.open('GET', url, true);
        xhr.onreadystatechange = function () {
            if (xhr.readyState === 4) {
                if (xhr.status >= 200 && xhr.status < 300) {
                    try {
                        var json = JSON.parse(xhr.responseText || '{}');
                        cb(null, json);
                    } catch (e) {
                        cb(e);
                    }
                } else {
                    cb(new Error('HTTP ' + xhr.status));
                }
            }
        };
        xhr.send(null);
    }

    function getToken(cb, attempt) {
        attempt = attempt || 1;
        if (_token) {
            cb(null, _token);
            return;
        }

        var url = resolveTokenUrl();
        log('Requesting socket token from', url, 'attempt', attempt);

        httpGetJson(url, function (err, json) {
            if (err || !json || !json.success || !json.data || !json.data.token) {
                if (err) errorLog('Token request failed:', err);
                else errorLog('Token response invalid:', json);

                if (attempt < 3) {
                    // simple retry with backoff
                    setTimeout(function () {
                        getToken(cb, attempt + 1);
                    }, attempt * 500);
                } else {
                    cb(err || new Error('Invalid token response'));
                }
                return;
            }

            _token = json.data.token;
            log('Received socket token for subject=' + (json.data.subject || 'n/a'));
            cb(null, _token);
        });
    }

    function drainConnectQueue(err, socket) {
        var queue = _connectQueue;
        _connectQueue = [];
        for (var i = 0; i < queue.length; i++) {
            try {
                queue[i](err, socket);
            } catch (e) {
                errorLog('Error in ensureConnected callback:', e);
            }
        }
    }

    // Local emit: calls our registered handlers regardless of socket
    function emitLocal(event, payload) {
        var list = _handlers[event];
        if (!list || !list.length) return;
        for (var i = 0; i < list.length; i++) {
            try {
                list[i](payload);
            } catch (e) {
                errorLog('Error in handler for event "' + event + '":', e);
            }
        }
    }

    
    var _boundEvents = null;
    // Bind all registered handlers to _socket, once per event
    function bindHandlersToSocket() {
        if (!_socket) return;

        if (!_boundEvents || _boundEvents.socketId !== _socket.id) {
            _boundEvents = { socketId: _socket.id, events: {} };
        }

        for (var event in _handlers) {
            if (!_handlers.hasOwnProperty(event) || event === 'connect') continue;
            if (_boundEvents.events[event]) continue; // already bound for this socket

            (function (ev) {
                _socket.on(ev, function (payload) {
                    emitLocal(ev, payload);
                });
            })(event);

            _boundEvents.events[event] = true;
        }
    }


    function connectInternal(cb) {
        if (!enabled) {
            cb(new Error('Socket is disabled by config.'));
            return;
        }

        if (typeof global.io === 'undefined') {
            cb(new Error('Socket.IO client (io) is not available. Make sure socket.io.min.js is loaded before dweb.socket.'));
            return;
        }

        if (_socket && _socket.connected) {
            cb(null, _socket);
            return;
        }

        _connectQueue.push(cb);

        if (_connecting) {
            return;
        }
        _connecting = true;

        getToken(function (err, token) {
            if (err) {
                _connecting = false;
                drainConnectQueue(err, null);
                return;
            }

            var url = resolveSocketUrl();
            log('Connecting to', url);

            try {
                _socket = global.io(url, {
                    auth: { token: token },
                    autoConnect: true
                });
            } catch (e) {
                _connecting = false;
                errorLog('Error creating socket.io client:', e);
                drainConnectQueue(e, null);
                return;
            }

            _socket.on('connect', function () {
                log('Socket connected. id=' + _socket.id);
                _connecting = false;

                // Attach event bridge(s) for non-connect events
                bindHandlersToSocket();

                // Fire logical 'connect' to our handlers
                emitLocal('connect', _socket);

                drainConnectQueue(null, _socket);
            });

            _socket.on('connect_error', function (e2) {
                errorLog('connect_error', e2 && e2.message ? e2.message : e2);
            });
            // Add near the top:
            var _lastAuthError = null;

            // In connectInternal(), update the auth_error listener:
            _socket.on('auth_error', function (payload) {
                errorLog('auth_error', payload);

                // If the server uses a specific code or message, you can branch on that.
                // For now, assume any auth_error means our token is bad/expired.
                _lastAuthError = payload || {};

                // Clear cached token so next connectInternal() will re-fetch
                _token = null;

                // Option A: proactively retry once after a short delay:
                setTimeout(function () {
                    if (!_socket || _socket.connected) return; // already fixed
                    _connecting = false; // allow connectInternal to run again
                    connectInternal(function (err) {
                        if (err) {
                            errorLog('Re-connect after auth_error failed:', err);
                        }
                    });
                }, 1000);
            });

            _socket.on('disconnect', function (reason) {
                log('Socket disconnected:', reason);
                // Fire logical 'disconnect' event through our own API
                emitLocal('disconnect', reason);
            });
        });
    }

    var api = {
        ensureConnected: function (cb) {
            if (typeof cb !== 'function') return;
            connectInternal(cb);
        },

        // Robust subscription:
        // - Stores handler in our registry.
        // - When connected, events are funneled from the underlying socket to emitLocal.
        // - 'connect' and 'disconnect' are special; see connectInternal().
        on: function (event, handler) {
            if (!event || typeof handler !== 'function') return;

            if (!_handlers[event]) {
                _handlers[event] = [];
            }

            // Avoid adding the exact same handler twice for the same event
            var list = _handlers[event];
            for (var i = 0; i < list.length; i++) {
                if (list[i] === handler) {
                    return; // already registered
                }
            }
            list.push(handler);

            // If already connected and this is not 'connect', bind bridge now
            if (_socket && _socket.connected && event !== 'connect') {
                _socket.on(event, function (payload) {
                    emitLocal(event, payload);
                });
            }
        },

        off: function (event, handler) {
            if (!event) return;

            var list = _handlers[event];
            if (!list || !list.length) return;

            // Remove handler from our registry
            if (typeof handler === 'function') {
                for (var i = list.length - 1; i >= 0; i--) {
                    if (list[i] === handler) {
                        list.splice(i, 1);
                    }
                }
            } else {
                // No handler -> remove all handlers for this event
                list.length = 0;
            }
        },


        emit: function (event, payload) {
            api.ensureConnected(function (err, socket) {
                if (err || !socket) {
                    errorLog('emit(' + event + ') failed:', err);
                    return;
                }
                socket.emit(event, payload);
            });
        },

        /**
         * Join a logical channel/room on the server.
         *
         * This is a convenience wrapper around socket.emit('join', {channel}).
         * It:
         *   - Ensures connection.
         *   - Emits a single join message.
         *   - Invokes cb(err, socket) when done (optional).
         *
         * Server side is responsible for interpreting the 'join' event.
         */
        join: function (channel, cb) {
            if (!channel || typeof channel !== 'string') {
                if (typeof cb === 'function') {
                    cb(new Error('Channel is required for join().'));
                }
                return;
            }

            api.ensureConnected(function (err, socket) {
                if (err || !socket) {
                    errorLog('join(' + channel + ') failed:', err);
                    if (typeof cb === 'function') {
                        cb(err || new Error('Socket not available'));
                    }
                    return;
                }

                socket.emit('join', { channel: channel });

                if (typeof cb === 'function') {
                    cb(null, socket);
                }
            });
        },

        disconnect: function () {
            if (_socket) {
                _socket.disconnect();
                _socket = null;
            }
            _token = null;
            _connecting = false;
            _connectQueue = [];
            _boundEvents = null;
        }

    };

    dweb.socket = api;

})(window);