// public/assets/dweb/dweb.xhr.js
(function (global) {
    'use strict';

    var dweb = global.dweb = global.dweb || {};
    var app  = dweb.app  = dweb.app  || {};

    /**
     * Low‑level helper to build query string from a plain object.
     */
    function buildQuery(params) {
        if (!params) return '';
        var parts = [];
        for (var key in params) {
            if (!Object.prototype.hasOwnProperty.call(params, key)) continue;
            var val = params[key];
            if (val === undefined || val === null) continue;
            parts.push(encodeURIComponent(key) + '=' + encodeURIComponent(val));
        }
        return parts.join('&');
    }

    /**
     * Append query parameters to a URL.
     */
    function appendQuery(url, params) {
        if (!params) return url;
        var qs = buildQuery(params);
        if (!qs) return url;
        return url + (url.indexOf('?') === -1 ? '?' : '&') + qs;
    }

    /**
     * POST application/x-www-form-urlencoded, expect JSON response.
     * Uses fetch if available, falls back to XMLHttpRequest.
     */
    function jsonPost(url, data) {
        var body = (data instanceof global.URLSearchParams)
            ? data.toString()
            : buildQuery(data || {});

        // Prefer fetch when available
        if (global.fetch) {
            return global.fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
                    'Accept': 'application/json'
                },
                body: body
            }).then(function (res) {
                return res.json();
            });
        }

        // Fallback: XHR (for older browsers)
        return new global.Promise(function (resolve, reject) {
            try {
                var xhr = new global.XMLHttpRequest();
                xhr.open('POST', url, true);
                xhr.setRequestHeader(
                    'Content-Type',
                    'application/x-www-form-urlencoded; charset=UTF-8'
                );
                xhr.onreadystatechange = function () {
                    if (xhr.readyState !== 4) return;
                    if (xhr.status >= 200 && xhr.status < 300) {
                        try {
                            resolve(JSON.parse(xhr.responseText));
                        } catch (e) {
                            reject(e);
                        }
                    } else {
                        reject(new Error('HTTP ' + xhr.status));
                    }
                };
                xhr.send(body);
            } catch (e) {
                reject(e);
            }
        });
    }

    /**
     * Generic fetch wrapper that:
     *  - accepts method/url/options
     *  - auto‑encodes body for JSON or form as requested
     *  - returns a promise for either .json() or .text()
     *
     * options:
     *   method        : 'GET' | 'POST' | ...
     *   headers       : { ... }
     *   body          : string | plain object | FormData | URLSearchParams
     *   responseType  : 'json' | 'text'
     */
    function fetchUrl(url, options) {
        options = options || {};
        var method = (options.method || 'GET').toUpperCase();
        var headers = options.headers || {};
        var body = options.body;
        var responseType = options.responseType || 'json'; // 'json' or 'text'

        // If a plain object body is passed, encode based on Content-Type
        if (body && typeof body === 'object' &&
            !(body instanceof global.FormData) &&
            !(body instanceof global.URLSearchParams)) {

            var ct = headers['Content-Type'] || headers['content-type'] || '';
            if (ct.indexOf('application/json') !== -1) {
                body = JSON.stringify(body);
            } else if (ct.indexOf('application/x-www-form-urlencoded') !== -1) {
                body = buildQuery(body);
            }
        }

        if (!global.fetch) {
            // Simple XHR fallback: only supports text / JSON, no streaming.
            return new global.Promise(function (resolve, reject) {
                try {
                    var xhr = new global.XMLHttpRequest();
                    xhr.open(method, url, true);
                    for (var h in headers) {
                        if (Object.prototype.hasOwnProperty.call(headers, h)) {
                            xhr.setRequestHeader(h, headers[h]);
                        }
                    }
                    xhr.onreadystatechange = function () {
                        if (xhr.readyState !== 4) return;
                        if (xhr.status >= 200 && xhr.status < 300) {
                            try {
                                if (responseType === 'text') {
                                    resolve(xhr.responseText);
                                } else {
                                    resolve(JSON.parse(xhr.responseText));
                                }
                            } catch (e) {
                                reject(e);
                            }
                        } else {
                            reject(new Error('HTTP ' + xhr.status));
                        }
                    };
                    xhr.send(body || null);
                } catch (e) {
                    reject(e);
                }
            });
        }

        // Native fetch path
        return global.fetch(url, {
            method: method,
            headers: headers,
            body: (method === 'GET' || method === 'HEAD') ? undefined : body
        }).then(function (res) {
            if (!res.ok) {
                var err = new Error('HTTP ' + res.status);
                err.response = res;
                throw err;
            }
            if (responseType === 'text') {
                return res.text();
            }
            return res.json();
        });
    }

    /**
     * Convenience: GET JSON with optional query params.
     */
    function getJSON(url, params, options) {
        options = options || {};
        options.responseType = 'json';
        options.method = 'GET';
        url = appendQuery(url, params || {});
        return fetchUrl(url, options);
    }

    /**
     * Convenience: generic GET, returns json (default) or text.
     */
    function get(url, params, options) {
        options = options || {};
        url = appendQuery(url, params || {});
        options.method = 'GET';
        options.responseType = options.responseType || 'json';
        return fetchUrl(url, options);
    }

    /**
     * Convenience: POST form-style data (x-www-form-urlencoded by default) and expect JSON.
     * If data is plain object, it will be encoded as x-www-form-urlencoded.
     */
    function postForm(url, data, options) {
        options = options || {};
        options.method = 'POST';
        options.headers = options.headers || {};

        if (!options.headers['Content-Type'] && !options.headers['content-type']) {
            options.headers['Content-Type'] = 'application/x-www-form-urlencoded; charset=UTF-8';
        }

        if (!options.body) {
            options.body = data;
        }

        options.responseType = options.responseType || 'json';
        return fetchUrl(url, options);
    }

    /**
     * Convenience: POST JSON body and expect JSON response.
     */
    function postJSON(url, bodyObj, options) {
        options = options || {};
        options.method = 'POST';
        options.headers = options.headers || {};

        if (!options.headers['Content-Type'] && !options.headers['content-type']) {
            options.headers['Content-Type'] = 'application/json; charset=UTF-8';
        }
        if (!options.headers['Accept'] && !options.headers['accept']) {
            options.headers['Accept'] = 'application/json';
        }

        options.body = JSON.stringify(bodyObj || {});
        options.responseType = options.responseType || 'json';

        return fetchUrl(url, options);
    }

    /**
     * File upload helper: POST FormData and parse JSON (default) or text.
     * If 'formData' is not a FormData, you can pass:
     *   { file: File, fields: { extra: 'value', ... } }
     */
    function upload(url, formData, options) {
        options = options || {};
        var responseType = options.responseType || 'json';

        // Normalize to FormData if user passed a helper object
        if (!(formData instanceof global.FormData)) {
            var fd = new global.FormData();
            if (formData && formData.file) {
                fd.append('file', formData.file);
            }
            if (formData && formData.fields) {
                for (var k in formData.fields) {
                    if (Object.prototype.hasOwnProperty.call(formData.fields, k)) {
                        fd.append(k, formData.fields[k]);
                    }
                }
            }
            formData = fd;
        }

        if (global.fetch && !options.forceXHR) {
            return global.fetch(url, {
                method: 'POST',
                body: formData
            }).then(function (res) {
                if (!res.ok) {
                    var err = new Error('HTTP ' + res.status);
                    err.response = res;
                    throw err;
                }
                return responseType === 'text' ? res.text() : res.json();
            });
        }

        // XHR fallback (also suitable if you later want to add progress callbacks)
        return new global.Promise(function (resolve, reject) {
            try {
                var xhr = new global.XMLHttpRequest();
                xhr.open('POST', url, true);

                xhr.onreadystatechange = function () {
                    if (xhr.readyState !== 4) return;
                    if (xhr.status >= 200 && xhr.status < 300) {
                        try {
                            if (responseType === 'text') {
                                resolve(xhr.responseText);
                            } else {
                                resolve(JSON.parse(xhr.responseText || 'null'));
                            }
                        } catch (e) {
                            reject(e);
                        }
                    } else {
                        reject(new Error('HTTP ' + xhr.status));
                    }
                };

                xhr.onerror = function () {
                    reject(new Error('Network error'));
                };

                xhr.send(formData);
            } catch (e) {
                reject(e);
            }
        });
    }

    /**
     * Thin jQuery‑style ajax wrapper to ease migration from $.ajax({...}).
     *
     * Supported subset:
     *   url        : string (required)
     *   type/method: 'GET' | 'POST' | ...
     *   dataType   : 'json' | 'text'
     *   data       : object (for GET query or POST body)
     *   contentType: 'application/json', etc. (for POST)
     *   headers    : extra headers
     *   success    : function(data, statusText, xhrLike)
     *   error      : function(xhrLike, statusText, error)
     *
     * Returns a Promise that resolves with the parsed data or rejects with Error.
     */
    function ajax(opts) {
        if (!opts || !opts.url) {
            throw new Error('app.xhr.ajax: url is required');
        }

        var method     = (opts.type || opts.method || 'GET').toUpperCase();
        var dataType   = (opts.dataType || 'json').toLowerCase();
        var url        = opts.url;
        var data       = opts.data || null;
        var success    = typeof opts.success === 'function' ? opts.success : null;
        var errorCb    = typeof opts.error   === 'function' ? opts.error   : null;
        var headers    = opts.headers || {};
        var contentType = opts.contentType || null;

        var promise;

        if (method === 'GET') {
            url = appendQuery(url, data);
            if (dataType === 'json') {
                promise = getJSON(url, null, { headers: headers });
            } else {
                promise = get(url, null, { headers: headers, responseType: 'text' });
            }
        } else {
            // Basic heuristic: if contentType is JSON, use postJSON;
            // else fall back to form-style POST.
            if (contentType && contentType.indexOf('application/json') !== -1) {
                var jsonOpts = { headers: headers };
                jsonOpts.headers['Content-Type'] = contentType;
                promise = postJSON(url, data, jsonOpts);
            } else {
                var formOpts = { headers: headers };
                if (contentType) {
                    formOpts.headers['Content-Type'] = contentType;
                }
                promise = postForm(url, data, formOpts);
            }
        }

        promise = promise.then(function (resData) {
            if (success) {
                // For now we pass null for xhrLike; you can later add a richer object.
                success(resData, 'success', null);
            }
            return resData;
        }).catch(function (err) {
            if (errorCb) {
                errorCb(null, 'error', err);
            }
            throw err;
        });

        return promise;
    }

    // Public API
    app.xhr = {
        // original exports (kept for backward compatibility)
        buildQuery: buildQuery,
        jsonPost: jsonPost,
        fetchUrl: fetchUrl,

        // new helpers
        appendQuery: appendQuery,
        get: get,
        getJSON: getJSON,
        postForm: postForm,
        postJSON: postJSON,
        upload: upload,
        ajax: ajax
    };

})(window);
