<?php
namespace boru\dweb\Assets;

use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Config\ConfigKeys;
use boru\dweb\Rendering\SectionBag;

class AssetManager
{
    const SECTION_HEAD    = 'head';
    const SECTION_SCRIPTS = 'scripts';

    /** @var SettingsInterface */
    private $settings;

    /** @var SectionBag */
    private $sections;

    /** @var string */
    private $basePrefix;

    /** @var string|null */
    private $version;

    public function __construct(SettingsInterface $settings, SectionBag $sections)
    {
        $this->settings = $settings;
        $this->sections = $sections;

        $prefix = (string)$settings->get(ConfigKeys::ASSET_URL_PREFIX, '/');
        $this->basePrefix = $this->normalizePrefix($prefix);

        $v = $settings->get(ConfigKeys::ASSET_VERSION, null);
        $this->version = ($v === null || $v === '') ? null : (string)$v;
    }

    /**
     * Build a public URL for an asset under the configured ASSET_URL_PREFIX.
     * Examples:
     *   prefix: /framework/  + path: assets/app.css => /framework/assets/app.css
     *   prefix: /           + path: assets/app.css => /assets/app.css
     */
    public function url($path)
    {
        $p = ltrim((string)$path, '/');
        $u = $this->basePrefix . $p;

        if ($this->version !== null) {
            $u .= (strpos($u, '?') === false ? '?' : '&') . 'v=' . rawurlencode($this->version);
        }

        return $u;
    }

    /** Register a stylesheet link into a section (defaults to head). */
    public function css($path, array $attrs = array(), $section = self::SECTION_HEAD)
    {
        $href = $this->url($path);

        $attrs = $this->mergeAttrs(array(
            'rel'  => 'stylesheet',
            'href' => $href,
        ), $attrs);

        $tag = '<link' . $this->attrsToHtml($attrs) . '>'."\n";
        $this->sections->append($section, $tag);

        return $tag;
    }

    /** Register a script tag into a section (defaults to scripts). */
    public function js($path, array $attrs = array(), $section = self::SECTION_SCRIPTS)
    {
        $src = $this->url($path);

        $attrs = $this->mergeAttrs(array(
            'src' => $src,
        ), $attrs);

        // Default type can be omitted in modern HTML, but explicit is fine.
        if (!isset($attrs['type'])) $attrs['type'] = 'text/javascript';

        $tag = '<script' . $this->attrsToHtml($attrs) . '></script>'."\n";
        $this->sections->append($section, $tag);

        return $tag;
    }

    /** Inline style block into a section (defaults to head). */
    public function inlineCss($cssText, $section = self::SECTION_HEAD, array $attrs = array())
    {
        $attrs = $this->mergeAttrs(array('type' => 'text/css'), $attrs);
        $tag = '<style' . $this->attrsToHtml($attrs) . '>' . (string)$cssText . '</style>'."\n";
        $this->sections->append($section, $tag);
        return $tag;
    }

    /** Inline script block into a section (defaults to scripts). */
    public function inlineJs($jsText, $section = self::SECTION_SCRIPTS, array $attrs = array())
    {
        $attrs = $this->mergeAttrs(array('type' => 'text/javascript'), $attrs);
        $tag = '<script' . $this->attrsToHtml($attrs) . '>' . (string)$jsText . '</script>'."\n";
        $this->sections->append($section, $tag);
        return $tag;
    }

    /** Render a section as HTML (head/scripts/etc). */
    public function render($section)
    {
        $html = $this->sections->get((string)$section, '');
        return $html === '' ? '' : ($html . "\n");
    }

    /** Access to SectionBag if you want advanced use. */
    public function sections()
    {
        return $this->sections;
    }

    // -------------------------
    // Internals
    // -------------------------

    private function normalizePrefix($prefix)
    {
        $prefix = (string)$prefix;
        if ($prefix === '') return '/';

        // Ensure leading slash
        if ($prefix[0] !== '/') $prefix = '/' . $prefix;

        // Ensure trailing slash
        $prefix = rtrim($prefix, '/') . '/';

        return $prefix;
    }

    private function mergeAttrs(array $base, array $overrides)
    {
        foreach ($overrides as $k => $v) {
            $base[(string)$k] = $v;
        }
        return $base;
    }

    private function attrsToHtml(array $attrs)
    {
        $out = '';
        foreach ($attrs as $k => $v) {
            if ($v === null) continue;

            $k = (string)$k;

            // boolean attribute support: ['defer' => true] => defer
            if ($v === true) {
                $out .= ' ' . $this->escAttr($k);
                continue;
            }

            $out .= ' ' . $this->escAttr($k) . '="' . $this->escAttr((string)$v) . '"';
        }
        return $out;
    }

    private function escAttr($s)
    {
        return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
    }
}
