<?php
namespace boru\dweb\Assets;

use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Contracts\LoggerInterface;
use boru\dweb\Contracts\AssetProviderInterface;
use boru\dweb\Kernel\ModuleManager;
use boru\dweb\Config\ConfigKeys;

class FilesystemAssetPublisher implements AssetPublisherInterface
{
    /** @var SettingsInterface */
    private $settings;

    /** @var ModuleManager */
    private $moduleManager;

    /** @var LoggerInterface */
    private $logger;

    public function __construct(SettingsInterface $settings, ModuleManager $moduleManager, LoggerInterface $logger)
    {
        $this->settings = $settings;
        $this->moduleManager  = $moduleManager;
        $this->logger   = $logger;
    }

    public function publishAll()
    {
        $res = new PublishResult();

        $enabled = (bool)$this->settings->get(ConfigKeys::ASSET_PUBLISH_ENABLED, false);
        if (!$enabled) {
            $res->addMessage('Asset publishing disabled (ConfigKeys::ASSET_PUBLISH_ENABLED).');
            return $res;
        }

        $publicDir = (string)$this->settings->get(ConfigKeys::ASSET_PUBLIC_DIR, '');
        if ($publicDir === '') {
            $res->errors++;
            $res->addMessage('Missing ConfigKeys::ASSET_PUBLIC_DIR (absolute path recommended).');
            return $res;
        }

        $publicDir = rtrim($publicDir, DIRECTORY_SEPARATOR);

        // Ensure base dir exists
        if (!is_dir($publicDir) && !@mkdir($publicDir, 0777, true)) {
            $res->errors++;
            $res->addMessage('Failed to create public assets dir: ' . $publicDir);
            return $res;
        }

        // ✅ 1) Publish core assets first (dweb.js, etc.)
        $coreDir = $this->coreAssetDir();
        if ($coreDir && is_dir($coreDir)) {
            $this->publishDir($coreDir, $publicDir, $res);
        } else {
            $res->addMessage('Core assets dir not found (skipped): ' . (string)$coreDir);
        }

        // ✅ 2) Publish module assets next
        foreach ($this->moduleManager->all() as $module) {
            if (!($module instanceof AssetProviderInterface)) {
                continue;
            }

            $moduleName = $module->getName();
            $srcDir = $module->getAssetPath();
            if ($srcDir === null || $srcDir === '') continue;

            $srcDir = rtrim($srcDir, DIRECTORY_SEPARATOR);
            if (!is_dir($srcDir)) {
                $res->skipped++;
                $res->addMessage("Skipped {$moduleName}: asset dir not found: {$srcDir}");
                continue;
            }

            $destDir = $publicDir . DIRECTORY_SEPARATOR . $moduleName;

            $this->logger->info("Publishing assets for {$moduleName}: {$srcDir} -> {$destDir}");
            $this->copyDir($srcDir, $destDir, $res);
        }

        return $res;
    }

    private function coreAssetDir()
    {
        // Optional override
        $cfg = $this->settings->get(ConfigKeys::ASSET_CORE_DIR, null);
        if ($cfg) return (string)$cfg;

        // Default: package resources/assets
        return rtrim(__DIR__, DIRECTORY_SEPARATOR)
            . DIRECTORY_SEPARATOR . '..'
            . DIRECTORY_SEPARATOR . '..'
            . DIRECTORY_SEPARATOR . 'resources'
            . DIRECTORY_SEPARATOR . 'assets';
    }

    private function publishDir($srcDir, $publicDir, PublishResult $res)
    {
        $srcDir = rtrim((string)$srcDir, DIRECTORY_SEPARATOR);
        $publicDir = rtrim((string)$publicDir, DIRECTORY_SEPARATOR);

        $it = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($srcDir, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($it as $file) {
            /** @var \SplFileInfo $file */
            $srcPath = $file->getPathname();

            // preserve structure under srcDir
            $rel = substr($srcPath, strlen($srcDir));
            $rel = ltrim(str_replace('\\', '/', $rel), '/');

            $destPath = $publicDir . DIRECTORY_SEPARATOR . str_replace('/', DIRECTORY_SEPARATOR, $rel);

            if ($file->isDir()) {
                if (!is_dir($destPath) && !@mkdir($destPath, 0775, true)) {
                    $res->errors++;
                    $res->addMessage('Failed to mkdir: ' . $destPath);
                }
                continue;
            }

            $destDir = dirname($destPath);
            if (!is_dir($destDir) && !@mkdir($destDir, 0775, true)) {
                $res->errors++;
                $res->addMessage('Failed to mkdir: ' . $destDir);
                continue;
            }

            if (!@copy($srcPath, $destPath)) {
                $res->errors++;
                $res->addMessage('Failed to copy: ' . $srcPath . ' -> ' . $destPath);
                continue;
            }

            $res->copied++;
        }
    }

    private function copyDir($src, $dest, PublishResult $res)
    {
        if (!is_dir($dest) && !@mkdir($dest, 0777, true)) {
            $res->errors++;
            $res->addMessage('Failed to create dir: ' . $dest);
            return;
        }

        $dh = @opendir($src);
        if (!$dh) {
            $res->errors++;
            $res->addMessage('Failed to open dir: ' . $src);
            return;
        }

        while (($name = readdir($dh)) !== false) {
            if ($name === '.' || $name === '..') continue;

            $srcPath  = $src  . DIRECTORY_SEPARATOR . $name;
            $destPath = $dest . DIRECTORY_SEPARATOR . $name;

            if (is_dir($srcPath)) {
                $this->copyDir($srcPath, $destPath, $res);
                continue;
            }

            // Copy if missing or changed by mtime/size
            $shouldCopy = true;
            if (is_file($destPath)) {
                $shouldCopy = (
                    @filesize($srcPath) !== @filesize($destPath)
                    || @filemtime($srcPath) > @filemtime($destPath)
                );
            }

            if (!$shouldCopy) {
                $res->skipped++;
                continue;
            }

            // Ensure dest dir exists
            $parent = dirname($destPath);
            if (!is_dir($parent) && !@mkdir($parent, 0777, true)) {
                $res->errors++;
                $res->addMessage('Failed to create parent dir: ' . $parent);
                continue;
            }

            if (!@copy($srcPath, $destPath)) {
                $res->errors++;
                $res->addMessage('Failed to copy: ' . $srcPath . ' -> ' . $destPath);
                continue;
            }

            $res->copied++;
        }

        closedir($dh);
    }
}
