<?php
namespace boru\dweb\Cli;

use boru\dweb\Support\KernelEnv;
use boru\dweb\Cli\Commands\HelpCommand;
use boru\dweb\Cli\Commands\PublishCommand;
use boru\dweb\Cli\Commands\SleepCommand;
use boru\dweb\Cli\Commands\SmokeTestCommand;
use boru\dweb\Cli\Commands\SocketServeCommand;
use boru\dweb\Support\ControllerContext;
use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Contracts\LoggerInterface;
use boru\dweb\Contracts\CliInvokerInterface;
use boru\dweb\Assets\AssetPublisherInterface;
use boru\dweb\Support\SocketTokenService;
use boru\dweb\Support\SocketPublisher;
use boru\dweb\Mvc\DwebHelper;


class App
{
    /** @var CommandRegistry */
    private $registry;

    /** @var ControllerContext|null */
    private $controllerContext;

    public function __construct(CommandRegistry $registry = null)
    {
        $this->registry = $registry ? $registry : new CommandRegistry();
    }

    /**
     * @param array $argv
     * @return int
     */
    public function run(array $argv)
    {
        $tokens = array_slice($argv, 1);

        // Pass 1: extract env flags anywhere in argv
        list($envArgs, $rest) = Args::extractEnv($tokens);

        // Determine command name from remaining tokens:
        // first non-flag token, otherwise "help"
        $cmdName = 'help';
        $cmdIndex = null;

        for ($i = 0; $i < count($rest); $i++) {
            $t = $rest[$i];

            // global help shortcut
            if ($t === '-h' || $t === '--help') {
                $cmdName = 'help';
                $cmdIndex = $i;
                break;
            }

            // first non-flag token is command name
            if (strlen($t) > 0 && $t[0] !== '-') {
                $cmdName = $t;
                $cmdIndex = $i;
                break;
            }
        }

        // args after the command token
        $cmdArgs = array();
        if ($cmdIndex !== null) {
            $cmdArgs = array_slice($rest, $cmdIndex + 1);
        } else {
            // no command token found; allow "help" with remaining flags
            $cmdArgs = $rest;
        }

        // Build env BEFORE registry if:
        // - command isn't "help", OR
        // - env flags were provided (so help can show module commands)
        $env = null;
        if ($cmdName !== 'help' || !$envArgs->isEmpty()) {
            try {
                $env = CliEnv::build($envArgs);
            } catch (\Exception $e) {
                $this->err("ERROR: " . $e->getMessage() . "\n\n");
                $this->err("Tip: dweb help\n");
                return 2;
            }
        }

        // Build registry with env-aware commands
        $this->registry = $this->buildRegistry($env);

        // Allow env/modules to extend registry (commands provided by modules)
        if ($env) {
            $this->maybeExtendRegistryFromEnv($env);

            // Build and inject ControllerContext for CLI commands that opt in
            $ctx = $this->buildControllerContextFromEnv($env);
            if ($ctx) {
                $this->controllerContext = $ctx;
                $this->registry->withControllerContext($ctx);
            }
        }

        // Resolve command
        $command = $this->registry->get($cmdName);
        if (!$command) {
            $this->err("Unknown command: {$cmdName}\n\n");
            $this->registry->get('help')->execute(array());
            return 2;
        }

        // Execute (CommandInterface remains execute(array $args))
        return $command->execute($cmdArgs);
    }

    /**
     * @param KernelEnv|null $env
     * @return CommandRegistry
     */
    private function buildRegistry($env)
    {
        $r = new CommandRegistry();

        $r->add(new HelpCommand($this));

        // Core commands can self-error if env is null (help still works)
        $r->add(new SmokeTestCommand($env));
        $r->add(new PublishCommand($env));

        // Socket serve command
        $r->add(new SocketServeCommand($env));

        $r->add(new SleepCommand());

        return $r;
    }

    /**
     * Optional: let host env register commands into a registry service.
     *
     * Pattern:
     *   container->set(CommandRegistry::class, $registry)
     *   modules add commands during register()/commands()
     */
    private function maybeExtendRegistryFromEnv(KernelEnv $env)
    {
        try {
            $c = $env->container();

            $extra = $c->get(\boru\dweb\Cli\CommandRegistry::class);
            if ($extra instanceof \boru\dweb\Cli\CommandRegistry) {
                foreach ($extra->all() as $cmd) {
                    $this->registry->add($cmd);
                }
            }
        } catch (\Exception $ignore) {
            // no-op
        }
    }

    /**
     * Build a ControllerContext suitable for CLI usage from KernelEnv.
     *
     * @param KernelEnv $env
     * @return ControllerContext|null
     */
    private function buildControllerContextFromEnv(KernelEnv $env)
    {
        $c = $env->container();

        // Settings are always present
        $settings = $c->get(SettingsInterface::class);

        // DwebHelper is cheap to construct: needs CanonicalUrl + module routes
        // You already have CanonicalUrl + routes in KernelBuilder.
        $canonical = $c->get(\boru\dweb\Routing\CanonicalUrl::class);
        $routes    = $env->routes();

        $dweb = new DwebHelper($canonical, "core");

        $ctx = new ControllerContext($settings, $dweb);

        // Optional extras if available in container
        if ($c->has(CliInvokerInterface::class)) {
            $ctx->withCli($c->get(CliInvokerInterface::class));
        }

        if ($c->has(LoggerInterface::class)) {
            $ctx->withLogger($c->get(LoggerInterface::class));
        }

        if ($c->has(AssetPublisherInterface::class)) {
            // If you prefer, you could pass a dedicated AssetManager here instead.
            // For now, give commands access to publisher via ctx()->assets().
            $assets = $c->get(AssetPublisherInterface::class);
            if ($assets instanceof \boru\dweb\Assets\AssetManager) {
                $ctx->withAssets($assets);
            }
        }

        if ($c->has(SocketTokenService::class)) {
            $ctx->withSocketTokenService($c->get(SocketTokenService::class));
        }

        if ($c->has(SocketPublisher::class)) {
            $ctx->withSocketPublisher($c->get(SocketPublisher::class));
        }

        // No userIdentity concept in CLI by default; host can extend if needed.

        return $ctx;
    }

    /** @return CommandInterface[] */
    public function listCommands()
    {
        return $this->registry->all();
    }

    public function out($s) { echo $s; }
    public function err($s) { fwrite(STDERR, $s); }
}
