<?php
namespace boru\dweb\Cli;

class Args
{
    /** @var array key => value|array */
    public $flags = array();

    /** @var array positional args */
    public $pos = array();

    /**
     * Parse args list (tokens).
     *
     * Supports:
     *  --k=v
     *  --k v
     *  --flag (boolean true)
     *  -k v   (single-letter)
     *  -abc   (treated as -a -b -c = true)
     */
    public static function parse(array $args)
    {
        $a = new self();

        for ($i = 0; $i < count($args); $i++) {
            $cur = $args[$i];

            // long flag
            if (substr($cur, 0, 2) === '--') {
                $eq = strpos($cur, '=');
                if ($eq !== false) {
                    $k = substr($cur, 2, $eq - 2);
                    $v = substr($cur, $eq + 1);
                    $a->addFlag($k, $v);
                    continue;
                }

                $k = substr($cur, 2);

                // --k <next> or --k (true)
                if (($i + 1) < count($args) && substr($args[$i + 1], 0, 1) !== '-') {
                    $a->addFlag($k, $args[$i + 1]);
                    $i++;
                } else {
                    $a->addFlag($k, true);
                }
                continue;
            }

            // short flag(s)
            if (substr($cur, 0, 1) === '-' && strlen($cur) > 1) {
                // -k value
                if (strlen($cur) === 2) {
                    $k = substr($cur, 1, 1);
                    if (($i + 1) < count($args) && substr($args[$i + 1], 0, 1) !== '-') {
                        $a->addFlag($k, $args[$i + 1]);
                        $i++;
                    } else {
                        $a->addFlag($k, true);
                    }
                    continue;
                }

                // -abc => -a -b -c
                for ($j = 1; $j < strlen($cur); $j++) {
                    $a->addFlag(substr($cur, $j, 1), true);
                }
                continue;
            }

            // positional
            $a->pos[] = $cur;
        }

        return $a;
    }

    /**
     * Extract env-affecting options from a full argv token list.
     *
     * Recognized:
     *  -e <file>
     *  --env <file>
     *  --env=<file>
     *  --env-bootstrap <file>
     *  --env-bootstrap=<file>
     *  --setEnv k=v
     *  --setEnv=k=v
     *
     * Returns: array(Args $env, array $restTokens)
     */
    public static function extractEnv(array $tokens)
    {
        $env = new self();
        $rest = array();

        $takeValueKeys = array(
            'e' => true,
            'env' => true,
            'env-bootstrap' => true,
            'setEnv' => true,
        );

        for ($i = 0; $i < count($tokens); $i++) {
            $cur = $tokens[$i];

            // -e <file>
            if ($cur === '-e') {
                $v = (($i + 1) < count($tokens)) ? $tokens[$i + 1] : null;
                if ($v !== null && substr($v, 0, 1) !== '-') {
                    // Normalize to env + env-bootstrap for compatibility
                    $env->addFlag('env', $v);
                    $env->addFlag('env-bootstrap', $v);
                    $i++;
                    continue;
                }
                // malformed -e, keep as rest
                $rest[] = $cur;
                continue;
            }

            // long: --k or --k=v
            if (substr($cur, 0, 2) === '--') {
                $eq = strpos($cur, '=');
                if ($eq !== false) {
                    $k = substr($cur, 2, $eq - 2);
                    $v = substr($cur, $eq + 1);

                    if (isset($takeValueKeys[$k])) {
                        if ($k === 'env') {
                            $env->addFlag('env', $v);
                            $env->addFlag('env-bootstrap', $v);
                        } else {
                            $env->addFlag($k, $v);
                        }
                        continue;
                    }

                    // not env-related
                    $rest[] = $cur;
                    continue;
                }

                $k = substr($cur, 2);
                if (isset($takeValueKeys[$k])) {
                    $v = (($i + 1) < count($tokens)) ? $tokens[$i + 1] : null;
                    if ($v !== null && substr($v, 0, 1) !== '-') {
                        if ($k === 'env') {
                            $env->addFlag('env', $v);
                            $env->addFlag('env-bootstrap', $v);
                        } else {
                            $env->addFlag($k, $v);
                        }
                        $i++;
                        continue;
                    }

                    // allow --setEnv as boolean? no, treat as malformed
                    $rest[] = $cur;
                    continue;
                }

                $rest[] = $cur;
                continue;
            }

            // everything else
            $rest[] = $cur;
        }

        return array($env, $rest);
    }

    /**
     * Get flag value (if repeated, returns the last).
     */
    public function get($key, $default = null)
    {
        if (!array_key_exists($key, $this->flags)) return $default;

        $v = $this->flags[$key];
        if (is_array($v)) {
            return count($v) ? $v[count($v) - 1] : $default;
        }
        return $v;
    }

    /**
     * Get all values for a possibly repeated flag.
     *
     * @return array
     */
    public function getAll($key)
    {
        if (!array_key_exists($key, $this->flags)) return array();

        $v = $this->flags[$key];
        return is_array($v) ? $v : array($v);
    }

    public function has($key)
    {
        return array_key_exists($key, $this->flags);
    }

    public function isEmpty()
    {
        return empty($this->flags) && empty($this->pos);
    }

    private function addFlag($key, $value)
    {
        $key = (string)$key;

        if (!array_key_exists($key, $this->flags)) {
            $this->flags[$key] = $value;
            return;
        }

        // convert to list on repeat
        if (!is_array($this->flags[$key])) {
            $this->flags[$key] = array($this->flags[$key]);
        }

        $this->flags[$key][] = $value;
    }
}
