<?php
namespace boru\dweb\Cli\Commands;

use boru\dweb\Cli\CommandInterface;
use boru\dweb\Cli\Args;
use boru\dweb\Support\KernelEnv;
use boru\dweb\Config\ConfigKeys;

/**
 * socket:serve
 *
 * Start the Node-based Socket.IO server using env + config.
 *
 * Usage:
 *   vendor/bin/dweb socket:serve --env-bootstrap=/abs/path/to/env.php [--port=NNNN]
 *
 * Behaviour:
 *   - Requires a KernelEnv (built via --env-bootstrap).
 *   - Reads config:
 *       dweb.socket.enabled
 *       dweb.socket.auth.mode
 *       dweb.socket.jwt.secret
 *       dweb.socket.node.bin
 *       dweb.socket.node.script
 *       dweb.socket.node.port
 *   - Resolves default Node script path to:
 *       <packageRoot>/node/dweb-socket-server.js
 *   - Exports:
 *       DWEB_SOCKET_AUTH_MODE
 *       DWEB_SOCKET_JWT_SECRET
 *   - Then execs:
 *       {nodeBin} {script} --port={port}
 *     streaming stdout/stderr via passthru().
 */
class SocketServeCommand implements CommandInterface
{
    /** @var KernelEnv|null */
    private $env;

    /**
     * @param KernelEnv|null $env
     */
    public function __construct(KernelEnv $env = null)
    {
        $this->env = $env;
    }

    /** @return string */
    public function name()
    {
        return 'socket:serve';
    }

    /** @return string */
    public function description()
    {
        return 'Start the Node-based Socket.IO server. Requires --env-bootstrap=...';
    }

    /**
     * @param array $args
     * @return int
     */
    public function execute(array $args)
    {
        if (!$this->env) {
            fwrite(STDERR, "ERROR: socket:serve requires env (use --env-bootstrap=...)\n");
            return 2;
        }

        $a = Args::parse($args);

        $config = $this->env->config();

        // 1) Check if socket support is enabled
        $enabled = (bool)$config->get(ConfigKeys::SOCKET_ENABLED, false);
        if (!$enabled) {
            fwrite(STDERR, "ERROR: " . ConfigKeys::SOCKET_ENABLED . " is false. Enable it in your env config.\n");
            return 2;
        }

        // 2) Determine auth mode + secret
        $authMode = (string)$config->get(ConfigKeys::SOCKET_AUTH_MODE, 'jwt');
        $authMode = $authMode !== '' ? $authMode : 'jwt';

        $jwtSecret = (string)$config->get(ConfigKeys::SOCKET_JWT_SECRET, '');
        if ($authMode === 'jwt' && $jwtSecret === '') {
            fwrite(STDERR,
                "ERROR: JWT auth mode requires " . ConfigKeys::SOCKET_JWT_SECRET . " to be set in env config.\n"
            );
            return 2;
        }

        // Optional: API secret for PHP -> Node publish bridge
        $apiSecret = (string)$config->get(ConfigKeys::SOCKET_API_SECRET, '');

        // 3) Resolve Node binary
        $nodeBin = (string)$config->get(ConfigKeys::SOCKET_NODE_BIN, 'node');
        if ($nodeBin === '') {
            $nodeBin = 'node';
        }

        // 4) Resolve Node server script path
        $script = $config->get(ConfigKeys::SOCKET_NODE_SCRIPT, null);
        if ($script === null || $script === '') {
            // Default: <packageRoot>/node/dweb-socket-server.js
            $script = $this->defaultNodeScriptPath();
        }

        if (!is_file($script)) {
            fwrite(STDERR, "ERROR: Node socket server script not found at: {$script}\n");
            fwrite(STDERR, "Tip: set " . ConfigKeys::SOCKET_NODE_SCRIPT . " in env config or create node/dweb-socket-server.js\n");
            return 2;
        }

        // 5) Determine port (CLI flag overrides config)
        $port = $a->get('port', null);
        if ($port === null || $port === true) {
            $port = $config->get(ConfigKeys::SOCKET_NODE_PORT, 3001);
        }
        $port = (int)$port;
        if ($port <= 0) {
            $port = 3001;
        }

        // 6) Prepare environment variables for the Node process
        $envVars = $this->buildEnvForProcess($authMode, $jwtSecret, $apiSecret);

        // 7) Build command line
        $cmd = $this->buildCommand($nodeBin, $script, $port);

        // 8) Info output
        fwrite(STDERR, "Starting socket server:\n");
        fwrite(STDERR, "  node bin : {$nodeBin}\n");
        fwrite(STDERR, "  script   : {$script}\n");
        fwrite(STDERR, "  port     : {$port}\n");
        $hasApiSecret = ($apiSecret !== '') ? 'yes' : 'no';
        fwrite(STDERR, "  apiSecret: {$hasApiSecret}\n");
        fwrite(STDERR, "  authMode : {$authMode}\n\n");


        // 9) Run the process (blocking, streams stdout/stderr)
        return $this->runProcess($cmd, $envVars);
    }

    /**
     * @return string
     */
    private function defaultNodeScriptPath()
    {
        // framework root is one level up from src/
        $root = dirname(__DIR__, 3);
        return $root . '/node/dweb-socket-server.js';
    }

    /**
     * Build environment variables array for the child process.
     *
     * @param string $authMode
     * @param string $jwtSecret
     * @param string $apiSecret
     * @return array
     */
    private function buildEnvForProcess($authMode, $jwtSecret, $apiSecret)
    {
        $env = $_ENV;

        // Fallback for PHP < 7.1 where $_ENV may be empty but getenv() works
        if (!is_array($env) || count($env) === 0) {
            $env = array();
        }

        $env['DWEB_SOCKET_AUTH_MODE'] = $authMode;

        if ($authMode === 'jwt' && $jwtSecret !== '') {
            $env['DWEB_SOCKET_JWT_SECRET'] = $jwtSecret;
        }

        if ($apiSecret !== '') {
            $env['DWEB_SOCKET_API_SECRET'] = $apiSecret;
        }

        return $env;
    }

    /**
     * @param string $nodeBin
     * @param string $script
     * @param int    $port
     * @return string
     */
    private function buildCommand($nodeBin, $script, $port)
    {
        // Very basic escaping – relying on typical POSIX shells.
        $nodeBinEsc = escapeshellcmd($nodeBin);
        $scriptEsc  = escapeshellarg($script);
        $portEsc    = escapeshellarg((string)$port);

        return $nodeBinEsc . ' ' . $scriptEsc . ' --port=' . $portEsc;
    }

    /**
     * Run the process via passthru(), setting env vars if possible.
     *
     * @param string $cmd
     * @param array  $envVars
     * @return int
     */
    private function runProcess($cmd, array $envVars)
    {
        // Prefer proc_open so we can control environment,
        // but fall back to passthru if unavailable or disabled.
        if (function_exists('proc_open')) {
            $descriptorspec = array(
                0 => array('pipe', 'r'),
                1 => array('pipe', 'w'),
                2 => array('pipe', 'w'),
            );

            $process = proc_open($cmd, $descriptorspec, $pipes, null, $envVars);

            if (!is_resource($process)) {
                fwrite(STDERR, "ERROR: Failed to start process: {$cmd}\n");
                return 2;
            }

            // Stream child stdout/stderr to our own
            stream_set_blocking($pipes[1], true);
            stream_set_blocking($pipes[2], true);

            while (!feof($pipes[1]) || !feof($pipes[2])) {
                $read = array();
                if (!feof($pipes[1])) $read[] = $pipes[1];
                if (!feof($pipes[2])) $read[] = $pipes[2];

                if (empty($read)) {
                    break;
                }

                $write = null;
                $except = null;
                $n = stream_select($read, $write, $except, 1);

                if ($n === false) {
                    break;
                }

                foreach ($read as $r) {
                    $data = fread($r, 8192);
                    if ($data === '' || $data === false) {
                        continue;
                    }
                    if ($r === $pipes[1]) {
                        echo $data;
                    } else {
                        fwrite(STDERR, $data);
                    }
                }
            }

            foreach ($pipes as $p) {
                if (is_resource($p)) fclose($p);
            }

            $status = proc_get_status($process);
            $exitCode = $status['exitcode'];

            proc_close($process);

            if ($exitCode === -1) {
                // Some platforms report -1; normalize to 1
                $exitCode = 1;
            }

            return (int)$exitCode;
        }

        // Fallback: set env into putenv() and use passthru
        foreach ($envVars as $k => $v) {
            putenv($k . '=' . $v);
        }

        passthru($cmd, $exitCode);
        return (int)$exitCode;
    }
}
