<?php
namespace boru\dweb\Cli;

use boru\dweb\Contracts\CliInvokerInterface;
use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Config\ConfigKeys;

class VendorBinDwebInvoker implements CliInvokerInterface
{
    /** @var SettingsInterface */
    private $cfg;

    public function __construct(SettingsInterface $cfg)
    {
        $this->cfg = $cfg;
    }

    public function run($command, array $args = array(), array $options = array())
    {
        $enabled = (bool)$this->cfg->get(ConfigKeys::CLI_ENABLED, true);
        if (!$enabled) {
            return new CliResult(2, '', 'CLI invocation is disabled (ConfigKeys::CLI_ENABLED=false).');
        }

        $detatch = !empty($options['detach']);
        if(isset($options['detach'])) {
            unset($options['detach']);
        }

        $bin = (string)$this->cfg->get(ConfigKeys::CLI_BIN_PATH, 'vendor/bin/dweb');
        $cwd = $this->cfg->get(ConfigKeys::CLI_CWD, null);
        $envBootstrap = $this->cfg->get(ConfigKeys::CLI_ENV_BOOTSTRAP, null);
        $timeout = $this->cfg->get(ConfigKeys::CLI_TIMEOUT_SEC, null);

        $binAbs = $this->resolveBinPath($bin);

        if (!file_exists($binAbs)) {
            $detailError = [
                'configured_path' => $bin,
                'resolved_path' => $binAbs,
                'cwd' => $cwd,
            ];
            return new CliResult(
                2,
                '',
                'CLI bin not found: ' . $binAbs . ' (set ' . ConfigKeys::CLI_BIN_PATH . ').. Details: ' . json_encode($detailError, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE)
            );
        }

        // Build argv tokens
        $argv = array();

        // Optional prefix (e.g. sudo or custom command)
        $prefix = $this->cfg->get(ConfigKeys::CLI_PREFIX, false);
        if ($prefix) {
            if ($prefix === true) {
                $prefix = 'sudo';
            }
            $prefix = (string)$prefix;
            if ($prefix !== '') {
                // Allow composite prefixes like "sudo -u www-data"
                $prefixTokens = preg_split('/\s+/', $prefix);
                foreach ($prefixTokens as $pt) {
                    if ($pt === '') continue;
                    $argv[] = $pt;
                }
            }
        }

        $argv[] = $binAbs;
        $argv[] = (string)$command;

        // Attach env bootstrap if configured and not already provided.
        if ($envBootstrap && !isset($options['env-bootstrap']) && !isset($options['env_bootstrap'])) {
            // normalize to CLI option name
            $options['env-bootstrap'] = $envBootstrap;
        }

        foreach ($options as $k => $v) {
            $k = (string)$k;
            $k = ltrim($k, '-'); // allow passing "--env-bootstrap" or "env-bootstrap"
            if ($v === false || $v === null) continue;

            if ($v === true) {
                $argv[] = '--' . $k;
            } else {
                $argv[] = '--' . $k . '=' . (string)$v;
            }
        }

        foreach ($args as $a) {
            $argv[] = (string)$a;
        }

        if( $detatch ) {
            return $this->execDetached($argv, $cwd);
        }
        return $this->exec($argv, $cwd, $timeout);
    }

    private function resolveBinPath($bin)
    {
        // If absolute, use it.
        if ($this->isAbsolutePath($bin)) return $bin;

        // Prefer configured cwd if present (treat as project root)
        $cwd = $this->cfg->get(ConfigKeys::CLI_CWD, null);
        if ($cwd) {
            $candidate1 = rtrim((string)$cwd, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $bin;
            if(file_exists($candidate1)) {
                return $candidate1;
            }
            //try 1 parent up.. just in case..
            $candidate2 = rtrim((string)$cwd, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . $bin;
            if(file_exists($candidate2)) {
                return $candidate2;
            }
            //try removing vendor prefix.. just in case..
            $candidate3 = rtrim((string)$cwd, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . str_replace('vendor' . DIRECTORY_SEPARATOR, '', $bin);
            if(file_exists($candidate3)) {
                return $candidate3;
            }
            //and try 1 level up for that too
            $candidate4 = rtrim((string)$cwd, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR .'..' . DIRECTORY_SEPARATOR . str_replace('vendor' . DIRECTORY_SEPARATOR, '', $bin);
            if(file_exists($candidate4)) {
                return $candidate4;
            }
            //fallback to candidate1
            return $candidate1; //fallback
        }

        // Fallback: relative to current working dir (web/cli runtime)
        return getcwd() . DIRECTORY_SEPARATOR . $bin;
    }

    private function isAbsolutePath($path)
    {
        $path = (string)$path;
        if ($path === '') return false;

        // Unix absolute
        if ($path[0] === '/' || $path[0] === '\\') return true;

        // Windows drive letter
        return (bool)preg_match('/^[A-Za-z]:[\\\\\\/]/', $path);
    }

    private function exec(array $argv, $cwd, $timeout)
    {
        $cmd = $this->argvToShellCommand($argv);

        $descriptors = array(
            0 => array('pipe', 'r'),
            1 => array('pipe', 'w'),
            2 => array('pipe', 'w'),
        );

        $process = @proc_open($cmd, $descriptors, $pipes, $cwd ? (string)$cwd : null);
        if (!is_resource($process)) {
            return new CliResult(2, '', 'Failed to start process via proc_open().');
        }

        // no stdin
        fclose($pipes[0]);

        stream_set_blocking($pipes[1], true);
        stream_set_blocking($pipes[2], true);

        $stdout = '';
        $stderr = '';

        // Optional timeout (best-effort; PHP 5.6 safe)
        $start = time();
        while (true) {
            $status = proc_get_status($process);
            if (!$status['running']) break;

            if ($timeout !== null) {
                $elapsed = time() - $start;
                if ($elapsed >= (int)$timeout) {
                    // terminate
                    @proc_terminate($process);
                    $stderr .= "\n[timeout] process terminated after {$elapsed}s\n";
                    break;
                }
            }

            // small sleep to avoid busy loop
            usleep(50000);
        }

        $stdout .= stream_get_contents($pipes[1]);
        $stderr .= stream_get_contents($pipes[2]);

        fclose($pipes[1]);
        fclose($pipes[2]);

        $exit = proc_close($process);

        // proc_close returns -1 sometimes; use last known status if possible
        if ($exit === -1) {
            $exit = isset($status['exitcode']) ? (int)$status['exitcode'] : 1;
        }

        return new CliResult($exit, $stdout, $stderr);
    }

    private function execDetached(array $argv, $cwd)
    {
        $cmd = $this->argvToShellCommand($argv);

        // If cwd is set, run inside it (avoid reliance on proc_open cwd for shell wrappers)
        // Use nohup + background + redirect stdin/out/err, and echo the PID.
        // Output: PID on stdout.
        //
        // Note: This is Linux/Unix oriented. For Windows, you'd need a different path.
        $cwd = $cwd ? (string)$cwd : null;

        $shell = $cmd;

        if ($cwd) {
            $shell = 'cd ' . escapeshellarg($cwd) . ' && ' . $shell;
        }

        // detach: stdin from /dev/null, stdout+stderr to /dev/null, background, print PID
        $shell = 'nohup sh -c ' . escapeshellarg($shell) . ' </dev/null >/dev/null 2>&1 & echo $!';

        $descriptors = array(
            0 => array('pipe', 'r'),
            1 => array('pipe', 'w'),
            2 => array('pipe', 'w'),
        );

        $proc = @proc_open($shell, $descriptors, $pipes);
        if (!is_resource($proc)) {
            return new CliResult(2, '', 'Failed to start detached process via proc_open().');
        }

        // We won't write to stdin
        fclose($pipes[0]);

        $pid = trim(stream_get_contents($pipes[1]));
        $err = trim(stream_get_contents($pipes[2]));

        fclose($pipes[1]);
        fclose($pipes[2]);

        $exit = proc_close($proc);

        if ($exit !== 0 || $pid === '') {
            // Best effort error detail
            $msg = $err ? $err : ('Detached start failed (exit=' . $exit . ').');
            return new CliResult(2, '', $msg);
        }

        return new CliResult(0, $pid, '');
    }


    private function argvToShellCommand(array $argv)
    {
        // escape each token; allow shebang execution of vendor/bin/dweb
        $out = array();
        foreach ($argv as $i => $token) {
            $t = (string)$token;
            // On windows you might want extra handling; for now keep it simple.
            $out[] = escapeshellarg($t);
        }
        return implode(' ', $out);
    }
}
