<?php
namespace boru\dweb\Http;

class Request
{
    /** @var array */
    private $query;
    /** @var array */
    private $post;
    /** @var array */
    private $server;
    /** @var array */
    private $cookies;
    /** @var array */
    private $files;
    /** @var array */
    private $headers;
    /** @var string|null */
    private $rawBody;
    /** @var array|null */
    private $jsonBody;
    /** @var array */
    private $meta;


    /**
     * @param array $query
     * @param array $post
     * @param array $server
     * @param array $cookies
     * @param array $files
     * @param array $headers
     * @param string|null $rawBody
     * @param array|null $jsonBody
     */
    public function __construct(
        array $query,
        array $post,
        array $server,
        array $cookies,
        array $files,
        array $headers,
        $rawBody = null,
        $jsonBody = null,
        array $meta = array()
    ) {
        $this->query    = $query;
        $this->post     = $post;
        $this->server   = $server;
        $this->cookies  = $cookies;
        $this->files    = $files;
        $this->headers  = $headers;
        $this->rawBody  = $rawBody;
        $this->jsonBody = $jsonBody;
        $this->meta     = $meta;
    }

    /**
     * Create from PHP globals.
     * Supports JSON bodies (application/json) by decoding php://input and merging into params.
     *
     * @return Request
     */
    public static function fromGlobals()
    {
        $server = isset($_SERVER) ? $_SERVER : array();
        $headers = self::extractHeaders($server);

        $raw = null;
        if (function_exists('file_get_contents')) {
            $raw = @file_get_contents('php://input');
            if ($raw === false) $raw = null;
        }

        $get  = isset($_GET) ? $_GET : array();
        $post = isset($_POST) ? $_POST : array();

        // Parse JSON body if present
        $jsonBody = null;
        $contentType = isset($headers['content-type']) ? (string)$headers['content-type'] : '';

        if ($raw !== null) {
            $trim = trim($raw);
            if ($trim !== '' && self::isJsonContentType($contentType)) {
                $decoded = json_decode($trim, true);
                if (is_array($decoded)) {
                    $jsonBody = $decoded;

                    // Merge JSON keys into POST params when POST doesn't already contain them.
                    // This allows param() to work transparently for JSON requests.
                    foreach ($decoded as $k => $v) {
                        if (!array_key_exists($k, $post)) {
                            $post[$k] = $v;
                        }
                    }
                }
            }
        }

        return new self(
            $get,
            $post,
            $server,
            isset($_COOKIE) ? $_COOKIE : array(),
            isset($_FILES) ? $_FILES : array(),
            $headers,
            $raw,
            $jsonBody,
            array()
        );
    }

    /**
     * Path portion of the request URI (no query string).
     * @return string
     */
    public function path()
    {
        $uri = (string)$this->server('REQUEST_URI', '/');
        $qpos = strpos($uri, '?');
        if ($qpos !== false) $uri = substr($uri, 0, $qpos);
        if ($uri === '') $uri = '/';
        return $uri;
    }

    /**
     * Return a new Request with additional query params merged in.
     * Does not mutate the existing request.
     *
     * @param array $params
     * @return Request
     */
    public function withQueryParams(array $params)
    {
        $q = $this->query;
        foreach ($params as $k => $v) {
            $q[(string)$k] = $v;
        }
        return new self(
            $q,
            $this->post,
            $this->server,
            $this->cookies,
            $this->files,
            $this->headers,
            $this->rawBody,
            $this->jsonBody,
            $this->meta
        );
    }

    /**
     * Return a new Request with an overridden HTTP method.
     * @param string $method
     * @return Request
     */
    public function withMethod($method)
    {
        $server = $this->server;
        $server['REQUEST_METHOD'] = strtoupper((string)$method);

        return new self(
            $this->query,
            $this->post,
            $server,
            $this->cookies,
            $this->files,
            $this->headers,
            $this->rawBody,
            $this->jsonBody,
            $this->meta
        );
    }

    public function withMeta($key, $value)
    {
        $m = $this->meta;
        $m[(string)$key] = $value;

        return new self(
            $this->query,
            $this->post,
            $this->server,
            $this->cookies,
            $this->files,
            $this->headers,
            $this->rawBody,
            $this->jsonBody,
            $m
        );
    }

    public function meta($key = null, $default = null)
    {
        if ($key === null) return $this->meta;
        $k = (string)$key;
        return array_key_exists($k, $this->meta) ? $this->meta[$k] : $default;
    }

    /** @return array */
    public function pathTail()
    {
        $v = $this->meta('dweb.path.tail', array());
        return is_array($v) ? $v : array();
    }

    /** @return array */
    public function pathParams()
    {
        $v = $this->meta('dweb.path.params', array());
        return is_array($v) ? $v : array();
    }

    public function pathParam($key, $default = null)
    {
        $p = $this->pathParams();
        $k = (string)$key;
        return array_key_exists($k, $p) ? $p[$k] : $default;
    }




    /**
     * Prefer POST over GET.
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function param($key, $default = null)
    {
        if (array_key_exists($key, $this->post)) return $this->post[$key];
        if (array_key_exists($key, $this->query)) return $this->query[$key];
        $p = $this->pathParams();
        if (array_key_exists($key, $p)) {
            return $p[$key];
        }
        return $default;
    }

    /**
     * Combined params (POST overrides GET).
     * @return array
     */
    public function all()
    {
        $all = $this->query;
        foreach ($this->post as $k => $v) $all[$k] = $v;
        return $all;
    }

    /** 
     * @param string|null $key
     * @param mixed $default
     * @return mixed|array
     */
    public function query($key=null,$default=null) { 
        if ($key === null) return $this->query;
        return array_key_exists($key, $this->query) ? $this->query[$key] : $default;
    }

    /**
     * @param string|null $key
     * @param mixed $default
     * @return mixed|array
     */
    public function post($key=null,$default=null) { 
        if ($key === null) return $this->post;
        return array_key_exists($key, $this->post) ? $this->post[$key] : $default;
    }

    /**
     * @param string|null $key
     * @param mixed $default
     * @return mixed|array
     */
    public function server($key=null,$default=null) { 
        if ($key === null) return $this->server;
        return array_key_exists($key, $this->server) ? $this->server[$key] : $default;
    }
    
    /**
     * @param string|null $key
     * @param mixed $default
     * @return mixed|array
     */
    public function cookies($key=null,$default=null) { 
        if ($key === null) return $this->cookies;
        return array_key_exists($key, $this->cookies) ? $this->cookies[$key] : $default;
    }
    
    /**
     * @param string|null $key
     * @param mixed $default
     * @return mixed|array
     */
    public function files($key=null,$default=null) { 
        if ($key === null) return $this->files;
        return array_key_exists($key, $this->files) ? $this->files[$key] : $default;
    }
    
    /** @return string */
    public function method()
    {
        $m = isset($this->server['REQUEST_METHOD']) ? $this->server['REQUEST_METHOD'] : 'GET';
        return strtoupper((string)$m);
    }

    /**
     * @param string $name
     * @param mixed $default
     * @return mixed
     */
    public function header($name, $default = null)
    {
        $k = strtolower($name);
        return isset($this->headers[$k]) ? $this->headers[$k] : $default;
    }

    /** @return array lowercase header map */
    public function headers() { return $this->headers; }

    /** @return string|null */
    public function rawBody() { return $this->rawBody; }

    /**
     * Parsed JSON body (if Content-Type: application/json and decode succeeded)
     * @return array|null
     */
    public function json()
    {
        return $this->jsonBody;
    }

    /**
     * Base path of the app mount, e.g. "/framework" or "" when at domain root.
     * Derived from SCRIPT_NAME (works for /framework/index.php).
     *
     * @return string
     */
    public function basePath()
    {
        $script = isset($_SERVER['SCRIPT_NAME']) ? (string)$_SERVER['SCRIPT_NAME'] : '';
        // remove trailing "/index.php" (or any script filename)
        $dir = rtrim(str_replace('\\', '/', dirname($script)), '/');

        // dirname("/") returns "\" on some setups, normalize
        if ($dir === '.' || $dir === '\\') $dir = '';
        if ($dir === '/') $dir = '';

        return $dir; // "" or "/framework"
    }

    /**
     * Prefix a path with basePath.
     * Accepts:
     *  - "Skeleton/home"
     *  - "/Skeleton/home" (leading slash ok)
     * Returns:
     *  - "/framework/Skeleton/home" when mounted, else "/Skeleton/home"
     *
     * @param string $path
     * @return string
     */
    public function url($path)
    {
        $path = (string)$path;
        if ($path === '') return $this->basePath() !== '' ? $this->basePath() . '/' : '/';

        // ensure single leading slash after basePath
        if ($path[0] !== '/') $path = '/' . $path;

        $base = $this->basePath();
        return $base !== '' ? ($base . $path) : $path;
    }

    // in boru\dweb\Http\Request

    /**
     * @return \boru\dweb\Contracts\SessionInterface|null
     */
    public function session()
    {
        $s = $this->meta('dweb.session', null);
        return $s instanceof \boru\dweb\Contracts\SessionInterface ? $s : null;
    }


    /**
     * @param string $contentType
     * @return bool
     */
    private static function isJsonContentType($contentType)
    {
        $contentType = strtolower((string)$contentType);
        // handles: application/json, application/json; charset=utf-8, etc.
        return (strpos($contentType, 'application/json') !== false) ||
               (strpos($contentType, '+json') !== false);
    }

    /**
     * Extract headers from $_SERVER in a PHP-agnostic way.
     * @param array $server
     * @return array
     */
    private static function extractHeaders(array $server)
    {
        $headers = array();

        if (function_exists('getallheaders')) {
            $h = @getallheaders();
            if (is_array($h)) {
                foreach ($h as $k => $v) {
                    $headers[strtolower($k)] = $v;
                }
                return $headers;
            }
        }

        foreach ($server as $key => $value) {
            if (strpos($key, 'HTTP_') === 0) {
                $name = strtolower(str_replace('_', '-', substr($key, 5)));
                $headers[$name] = $value;
            } elseif ($key === 'CONTENT_TYPE') {
                $headers['content-type'] = $value;
            } elseif ($key === 'CONTENT_LENGTH') {
                $headers['content-length'] = $value;
            } elseif ($key === 'CONTENT_MD5') {
                $headers['content-md5'] = $value;
            }
        }

        return $headers;
    }
}
