<?php
namespace boru\dweb\Http;

/**
 * SSE response built on top of the core Response streaming support.
 *
 * Usage:
 *   return SseResponse::stream(function($send) {
 *      $send(['tick' => 1], 'tick', '1');
 *   });
 */
class SseResponse extends Response
{
    /**
     * Create a streaming SSE response.
     *
     * Callback signature:
     *   function(callable $send): void
     *
     * Where $send supports:
     *   $send($data, $event = null, $id = null, $retry = null)
     *   $send(SseEvent $event)
     *
     * @param callable $callback
     * @param array $headers
     * @param int $status
     * @return Response
     */
    public static function stream($callback, array $headers = array(), $status = 200)
    {
        $base = self::defaultHeaders();
        foreach ($headers as $k => $v) $base[$k] = $v;

        // Use core streaming response
        return Response::stream(function () use ($callback) {

            // SSE connections can be long-lived.
            if (function_exists('set_time_limit')) {
                @set_time_limit(0);
            }

            $send = function ($data, $event = null, $id = null, $retry = null) {

                // Allow passing an event object directly
                if ($data instanceof SseEvent) {
                    echo $data->toWire();
                } else {
                    $e = new SseEvent($data, $event, $id, $retry);
                    echo $e->toWire();
                }

                // Important for SSE: push bytes
                if (function_exists('flush')) @flush();
            };

            // Initial comment keeps some proxies happy
            echo ": dweb-sse\n\n";
            if (function_exists('flush')) @flush();

            call_user_func($callback, $send);

        },$base, $status);
    }

    /**
     * Create a non-streaming SSE response (rare, but useful for tests).
     *
     * @param SseEvent[] $events
     * @param array $headers
     * @param int $status
     * @return Response
     */
    public static function events(array $events, array $headers = array(), $status = 200)
    {
        $base = self::defaultHeaders();
        foreach ($headers as $k => $v) $base[$k] = $v;

        $out = '';
        foreach ($events as $e) {
            if ($e instanceof SseEvent) $out .= $e->toWire();
        }

        return new Response($status, $base, $out);
    }

    private static function defaultHeaders()
    {
        return array(
            'Content-Type' => 'text/event-stream; charset=utf-8',
            'Cache-Control' => 'no-cache',
            'Connection' => 'keep-alive',
            // nginx buffering off
            'X-Accel-Buffering' => 'no',
        );
    }
}
