<?php
namespace boru\dweb\Modules\Core\Actions;

use boru\dweb\Mvc\AbstractAction;
use boru\dweb\Http\Request;
use boru\dweb\Support\SocketTokenService;
use boru\dweb\Config\ConfigKeys;
use boru\dweb\Domain\UserIdentity;

/**
 * Core infrastructure endpoint: issue a socket auth token.
 *
 * URL:
 *   ?action=core.socketToken
 *
 * Behavior:
 *   - Uses SocketTokenService to issue JWT/opaque token.
 *   - Determines subject from:
 *       1) Current UserIdentity (if available in context), OR
 *       2) ?userId=... query param, OR
 *       3) "anonymous" as a last resort (for demos).
 *
 * Response:
 *   { "success": true, "data": { "token": ..., "mode": "jwt", "ttl": 3600, "subject": "..." } }
 *
 * Host apps that need stricter semantics can:
 *   - Implement their own action, OR
 *   - Wrap this in additional auth checks.
 */
class SocketTokenAction extends AbstractAction
{
    public function handle(Request $req)
    {
        $settings = $this->settings();
        $ctx      = $this->ctx();

        // 1) Prefer a UserIdentity from context if available
        $subject = null;
        $identity = null;

        if (method_exists($ctx, 'userIdentity')) {
            // Some hosts may expose a UserIdentity here
            $identity = $ctx->userIdentity();
        }

        if ($identity !== null) {
            $methodAndPropsCheck = [
                "id",
                "identity",
                "subject"
            ];

            if(is_object($identity)) {
                //object identity
                foreach ($methodAndPropsCheck as $check) {
                    if (method_exists($identity, $check)) {
                        $subject = (string)$identity->$check();
                        break;
                    } elseif (property_exists($identity, $check)) {
                        $subject = (string)$identity->$check;
                        break;
                    }
                }
            } elseif(is_array($identity)) {
                //array identity
                foreach ($methodAndPropsCheck as $check) {
                    if (array_key_exists($check, $identity)) {
                        $subject = (string)$identity[$check];
                        break;
                    }
                }
            } elseif(is_string($identity)) {
                //string identity
                $subject = $identity;
            }
        }

        // 2) Allow explicit override via ?userId=... (useful for demos/testing)
        $explicitUserId = $req->param('userId', null);
        if ($explicitUserId !== null && $explicitUserId !== '') {
            $subject = (string)$explicitUserId;
        }

        // 3) Fallback
        if ($subject === null || $subject === '') {
            $subject = 'anonymous';
        }

        $mode = (string)$settings->get(ConfigKeys::SOCKET_AUTH_MODE, SocketTokenService::MODE_JWT);
        if ($mode === '') {
            $mode = SocketTokenService::MODE_JWT;
        }

        $ttl = (int)$settings->get(ConfigKeys::SOCKET_TOKEN_TTL, 3600);
        if ($ttl <= 0) {
            $ttl = 3600;
        }

        $tokens = $this->ctx()->socketTokenService();
        if ($tokens === null) {
            return $this->error(
                'SocketTokenService not available',
                500
            );
        }

        try {
            $token = $tokens->issueToken($subject, array(
                // Minimal standard claims beyond sub/iat/exp:
                'mode' => $mode,
            ));
        } catch (\Exception $e) {
            return $this->error(
                'Failed to issue socket token: ' . $e->getMessage(),
                500
            );
        }

        return $this->success(array(
            'token'   => $token,
            'mode'    => $mode,
            'ttl'     => $ttl,
            'subject' => $subject,
        ));
    }
}
