<?php
namespace boru\dweb\Modules\Realtime\Actions;

use boru\dweb\Mvc\AbstractAction;
use boru\dweb\Http\Request;
use boru\dweb\Support\SocketTokenService;
use boru\dweb\Config\ConfigKeys;

/**
 * Issue a socket auth token for the current user (or a demo user).
 *
 * Response:
 *   {
 *     "success": true,
 *     "data": {
 *       "token": "....",
 *       "mode": "jwt" | "opaque",
 *       "ttl": 3600,
 *       "subject": "user-123"
 *     }
 *   }
 */
class SocketTokenAction extends AbstractAction
{
    public function handle(Request $req)
    {
        $settings = $this->settings();

        // In a real app, derive subject from authenticated user identity.
        // For demo, accept ?userId=... or fall back to "demo-user".
        $userId = $req->param('userId', 'demo-user');
        $userId = (string)$userId;

        $mode = (string)$settings->get(ConfigKeys::SOCKET_AUTH_MODE, SocketTokenService::MODE_JWT);
        if ($mode === '') {
            $mode = SocketTokenService::MODE_JWT;
        }

        $ttl = (int)$settings->get(ConfigKeys::SOCKET_TOKEN_TTL, 3600);
        if ($ttl <= 0) {
            $ttl = 3600;
        }

        $tokens = $this->ctx()->socketTokenService();
        if ($tokens === null) {
            return $this->error(
                'SocketTokenService not available',
                500
            );
        }

        try {
            $token = $tokens->issueToken($userId, array(
                // Additional claims; kept small for demo
                'role' => 'demo',
            ));
        } catch (\Exception $e) {
            return $this->error(
                'Failed to issue socket token: ' . $e->getMessage(),
                500
            );
        }

        return $this->success(array(
            'token'   => $token,
            'mode'    => $mode,
            'ttl'     => $ttl,
            'subject' => $userId,
        ));
    }
}
