<?php
namespace boru\dweb\Mvc;

use boru\dweb\Contracts\CliInvokerInterface;
use boru\dweb\Contracts\RendererInterface;
use boru\dweb\Http\DWebHeaders;
use boru\dweb\Http\HtmlResponse;
use boru\dweb\Http\HxResponse;
use boru\dweb\Rendering\SectionBag;
use boru\dweb\Contracts\ControllerContextInterface;
use boru\dweb\Exceptions\Fail;

abstract class AbstractView
{
    /** @var ControllerContextInterface|null */
    private $ctx;

    /** @var RendererInterface */
    protected $renderer;

    /** @var string */
    protected $moduleName;

    /** @var SectionBag|null */
    private $sections;

    /** @var DwebHelper */
    protected $dweb;

    /** @var CliInvokerInterface */
    private $cli;

    /**
     * @param RendererInterface $renderer
     * @param string $moduleName
     */
    public function __construct(RendererInterface $renderer, $moduleName)
    {
        $this->renderer = $renderer;
        $this->moduleName = (string)$moduleName;
    }

    public function setContext(ControllerContextInterface $ctx)
    {
        $this->ctx = $ctx;
    }

    /** @return ControllerContextInterface */
    protected function ctx()
    {
        if (!$this->ctx) {
            throw Fail::runtime('ControllerContext not set for controller: ' . get_class($this),["controller"=>get_class($this)]);
        }
        return $this->ctx;
    }

    /**
     * Optional injection point (used by default factories).
     * Does not require constructor signature changes.
     *
     * @param DwebHelper $dweb
     * @return void
     */
    public function setDwebHelper(DwebHelper $dweb)
    {
        $this->dweb = $dweb;
    }
    
    /**
     * Controller-side access to canonical urls + section conventions.
     * @return DwebHelper
     */
    protected function dweb()
    {
        if (!$this->dweb) {
            try {
                $this->dweb = $this->ctx()->dweb();
            } catch (\RuntimeException $e) {
                throw Fail::runtime('DwebHelper not provided to controller: ' . get_class($this),["controller"=>get_class($this)]);
            }
        }
        return $this->dweb;
    }

    public function setCliInvoker(CliInvokerInterface $cli)
    {
        $this->cli = $cli;
    }

    protected function cli()
    {
        if (!$this->cli) {
            try {
                $this->cli = $this->ctx()->cli();
            } catch (\RuntimeException $e) {
                throw Fail::runtime('CliInvoker not set for controller: ' . get_class($this),["controller"=>get_class($this)]);
            }
        }
        return $this->cli;
    }

    protected function settings()
    {
        return $this->ctx()->settings();
    }

    protected function logger()
    {
        return $this->ctx()->logger();
    }

    protected function assets()
    {
        return $this->ctx()->assets();
    }

    /**
     * Forward server-side to another *view* (same or different module) using a qualified name.
     *
     * Example:
     *   return $this->forwardView('OtherModule.List');
     *
     * @param string $qualifiedView "Module.View"
     * @param \boru\dweb\Http\Request|null $reqOverride Optional Request override
     * @return object Response (typically HtmlResponse)
     */
    protected function forwardView($qualifiedView, \boru\dweb\Http\Request $reqOverride = null)
    {
        $ctx = $this->ctx();
        return $ctx->dispatchView((string)$qualifiedView, $reqOverride);
    }

    /**
     * Forward server-side to another *action* (same or different module) using a qualified name.
     *
     * Example:
     *   return $this->forwardAction('OtherModule.Save');
     *
     * @param string $qualifiedAction "Module.Action"
     * @param \boru\dweb\Http\Request|null $reqOverride Optional Request override
     * @return object Response (typically JsonResponse or HtmlResponse)
     */
    protected function forwardAction($qualifiedAction, \boru\dweb\Http\Request $reqOverride = null)
    {
        $ctx = $this->ctx();
        return $ctx->dispatchAction((string)$qualifiedAction, $reqOverride);
    }

    /**
     * Convenience: forward to another view by module + view.
     * If $moduleName is null/empty, use the current controller module.
     *
     * @param string|null $moduleName
     * @param string $viewName
     * @param \boru\dweb\Http\Request|null $reqOverride
     * @return object
     */
    protected function gotoView($moduleName, $viewName, \boru\dweb\Http\Request $reqOverride = null)
    {
        if ($moduleName === null || $moduleName === '') {
            $moduleName = $this->moduleName;
        }
        $qualified = $moduleName . '.' . (string)$viewName;
        return $this->forwardView($qualified, $reqOverride);
    }

    /**
     * Convenience: forward to another action by module + action.
     * If $moduleName is null/empty, use the current controller module.
     *
     * @param string|null $moduleName
     * @param string $actionName
     * @param \boru\dweb\Http\Request|null $reqOverride
     * @return object
     */
    protected function gotoAction($moduleName, $actionName, \boru\dweb\Http\Request $reqOverride = null)
    {
        if ($moduleName === null || $moduleName === '') {
            $moduleName = $this->moduleName;
        }
        $qualified = $moduleName . '.' . (string)$actionName;
        return $this->forwardAction($qualified, $reqOverride);
    }

    /**
     * Render a template without layout.
     *
     * @param string $templateId
     * @param array $data
     * @param int $status
     * @param array $headers
     * @return HtmlResponse
     */
    protected function html($templateId, array $data = array(), $status = 200, array $headers = array())
    {
        $html = $this->renderer->render($this->moduleName, $templateId, $data);
        return new HtmlResponse($html, $status, $headers);
    }

    /**
     * Render a template inside a layout (explicit).
     *
     * @param string $layoutId
     * @param string $templateId
     * @param array $data
     * @param int $status
     * @param array $headers
     * @return HtmlResponse
     */
    protected function layout($layoutId, $templateId, array $data = array(), $status = 200, array $headers = array())
    {
        $html = $this->renderer->renderLayout($this->moduleName, $layoutId, $templateId, $data);
        return new HtmlResponse($html, $status, $headers);
    }

    /**
     * Semantic alias for html(): indicates intent that this is a partial render.
     */
    protected function partial($templateId, array $data = array(), $status = 200, array $headers = array())
    {
        return $this->html($templateId, $data, $status, $headers);
    }

    /**
     * Render a fragment (no layout) from templates/fragments/<id>.
     * Tags the response for downstream layers (e.g., HTMX) without magic.
     */
    protected function fragment($fragmentId, array $data = array(), $status = 200, array $headers = array())
    {
        $headers['X-DWeb-Fragment'] = $this->moduleName . '.' . (string)$fragmentId;
        return $this->html('fragments/' . (string)$fragmentId, $data, $status, $headers);
    }

    /**
     * Explicit fragment response with optional HTMX headers.
     *
     * $hx is an associative array of instructions, e.g.:
     *  array(
     *    'retarget' => '#content',
     *    'reswap'   => 'outerHTML',
     *    'pushUrl'  => '/Skeleton/home',
     *    'trigger'  => array('toast' => array('msg' => 'Saved!')),
     *  )
     *
     * @param string $fragmentId
     * @param array $data
     * @param array $hx
     * @param int $status
     * @param array $headers
     * @return \boru\dweb\Http\HtmlResponse
     */
    protected function fragmentHx($fragmentId, array $data = array(), array $hx = array(), $status = 200, array $headers = array())
    {
        $res = $this->fragment($fragmentId, $data, $status, $headers);
        return $this->applyHx($res, $hx);
    }

    /**
     * Apply HTMX headers to any HtmlResponse explicitly.
     * @param \boru\dweb\Http\HtmlResponse $res
     * @param array $hx
     * @return \boru\dweb\Http\HtmlResponse
     */
    protected function applyHx($res, array $hx)
    {
        if (isset($hx['redirect']))  HxResponse::redirect($res, $hx['redirect']);
        if (isset($hx['refresh']))   HxResponse::refresh($res, (bool)$hx['refresh']);
        if (isset($hx['location']))  HxResponse::location($res, $hx['location']);
        if (isset($hx['pushUrl']))   HxResponse::pushUrl($res, $hx['pushUrl']);
        if (isset($hx['replaceUrl']))HxResponse::replaceUrl($res, $hx['replaceUrl']);
        if (isset($hx['reswap']))    HxResponse::reswap($res, $hx['reswap']);
        if (isset($hx['retarget']))  HxResponse::retarget($res, $hx['retarget']);

        if (isset($hx['trigger']))          HxResponse::trigger($res, $hx['trigger'], 'now');
        if (isset($hx['triggerAfterSwap'])) HxResponse::trigger($res, $hx['triggerAfterSwap'], 'afterSwap');
        if (isset($hx['triggerAfterSettle'])) HxResponse::trigger($res, $hx['triggerAfterSettle'], 'afterSettle');

        // optional explicit fragment tag override (mostly for non-view cases)
        if (isset($hx['fragment'])) HxResponse::fragment($res, $hx['fragment']);

        return $res;
    }

    /**
     * Lazily create / reuse section bag for this view invocation.
     * @return SectionBag
     */
    protected function sections()
    {
        if ($this->sections === null) {
            $this->sections = new SectionBag();
        }
        return $this->sections;
    }

    /**
     * Replace a section explicitly.
     * @param string $name
     * @param string $html
     * @return void
     */
    protected function sectionSet($name, $html)
    {
        $this->sections()->set($name, $html);
    }

    /**
     * Append to a section explicitly.
     * @param string $name
     * @param string $html
     * @return void
     */
    protected function sectionAppend($name, $html)
    {
        $this->sections()->append($name, $html);
    }

    /**
     * Render a template to a string (no layout), intended for sections.
     * This is explicit "partial render".
     *
     * @param string $templateId
     * @param array $data
     * @return string
     */
    protected function renderPartialToString($templateId, array $data = array())
    {
        return $this->renderer->render($this->moduleName, (string)$templateId, $data);
    }

    /**
     * Common pattern:
     * - render your main view template into "content"
     * - render sidebar/header/etc into other sections
     * - render layout with $sections available
     *
     * @param string $viewTemplateId e.g. "home/index"
     * @param array $data
     * @param string $layoutId e.g. "layout/main" (whatever your renderer expects)
     * @param int $status
     * @param array $headers
     * @return HtmlResponse
     */
    protected function htmlWithSections($viewTemplateId, array $data, $layoutId, $status = 200, array $headers = array())
    {
        $content = $this->renderer->render($this->moduleName, (string)$viewTemplateId, $data);

        // Ensure a default "content" section exists
        $sections = $this->sections();
        $sections->ensure('content', $content);

        // Layout gets both:
        // - $content (back-compat / simplest)
        // - $sections (new slot system)
        $layoutData = $data;
        $layoutData['content']  = $sections->get('content');
        $layoutData['sections'] = $sections->all();

        // Assumes you already have a layout render method in renderer:
        // renderLayout($moduleName, $layoutId, $layoutData)
        $html = $this->renderer->render($this->moduleName, (string)$layoutId, $layoutData);

        return new HtmlResponse($html, $status, $headers);
    }

    /**
     * Default section selector convention used by this view.
     * Override in a child view if you want a different convention.
     *
     * @param string $sectionName
     * @return string
     */
    protected function sectionSelector($sectionName)
    {
        return '#section-' . (string)$sectionName;
    }

    /**
     * Fragment -> Section targeting (explicit + convenient):
     * - Always annotates X-DWeb-Section
     * - Optionally auto-applies HX-Retarget/HX-Reswap using the section selector convention
     * - Caller can override by providing 'retarget'/'reswap' in $hx or custom $selector/$swap
     *
     * @param string $sectionName
     * @param string $fragmentId
     * @param array  $data
     * @param array  $hx
     * @param int    $status
     * @param array  $headers
     * @param string|null $selector If null, uses sectionSelector($sectionName)
     * @param string|null $swap If null, defaults to 'outerHTML'
     * @param bool $applyHxTargeting If true, will set HX headers when not already set
     * @return \boru\dweb\Http\HtmlResponse
     */
    protected function fragmentToSection(
        $sectionName,
        $fragmentId,
        array $data = array(),
        array $hx = array(),
        $status = 200,
        array $headers = array(),
        $selector = null,
        $swap = null,
        $applyHxTargeting = true
    ) {
        $sectionName = (string)$sectionName;

        // Always annotate which section this fragment belongs to
        $headers[DWebHeaders::SECTION] = $sectionName;

        // Convention selector + default swap
        if ($selector === null) $selector = $this->sectionSelector($sectionName);
        if ($swap === null) $swap = 'outerHTML';

        // Optional: include generic hints for any clients
        $headers[DWebHeaders::SELECTOR] = (string)$selector;
        $headers[DWebHeaders::SWAP]     = (string)$swap;

        // Optional: apply HTMX targeting headers (still explicit, but convenient)
        if ($applyHxTargeting) {
            if (!isset($hx['retarget'])) $hx['retarget'] = (string)$selector;
            if (!isset($hx['reswap']))   $hx['reswap']   = (string)$swap;
        }

        // Leverage your existing fragmentHx() (already tags fragment + HX headers)
        return $this->fragmentHx($fragmentId, $data, $hx, $status, $headers);
    }

    protected function sse($callback, array $headers = array(), $status = 200)
    {
        return \boru\dweb\Http\SseResponse::stream($callback, $headers, $status);
    }

    /**
     * Render a template into a named section in one step.
     *
     * @param string $sectionName
     * @param string $templateId
     * @param array  $data
     * @return void
     */
    protected function sectionTemplate($sectionName, $templateId, array $data = array())
    {
        $html = $this->renderPartialToString($templateId, $data);
        $this->sectionSet($sectionName, $html);
    }

    /**
     * Render a layout using any sections that have been configured
     * via sectionSet()/sectionAppend()/sectionTemplate().
     *
     * If 'content' is not already set, an empty string is used.
     *
     * @param string $layoutId
     * @param int    $status
     * @param array  $headers
     * @return HtmlResponse
     */
    protected function layoutWithSections($layoutId, $status = 200, array $headers = array())
    {
        $sections = $this->sections();

        // Ensure at least a content key exists (empty is okay)
        $sections->ensure('content', '');

        $layoutData = array();
        $layoutData['content']  = $sections->get('content');
        $layoutData['sections'] = $sections->all();

        // NEW: go through renderer’s layout resolution path
        $html = $this->renderer->renderLayoutWithSections(
            $this->moduleName,
            (string)$layoutId,
            $layoutData
        );

        return new HtmlResponse($html, $status, $headers);
    }

    /**
     * Render a layout with structured array sets for sections layoutWithSections is easier to use.
     * @param mixed $layoutId 
     * @param array $spec 
     * @param int $status 
     * @param array $headers 
     * @return HtmlResponse 
     */
    protected function layoutSections($layoutId, array $spec, $status = 200, array $headers = array())
    {
        foreach ($spec as $sectionName => $config) {
            if (!is_array($config) || !isset($config['template'])) continue;

            $data = isset($config['data']) && is_array($config['data']) ? $config['data'] : array();
            $this->sectionTemplate($sectionName, $config['template'], $data);
        }

        return $this->layoutWithSections($layoutId, $status, $headers);
    }

}
