<?php
namespace boru\dweb\Rendering;

use boru\dweb\Contracts\RendererInterface;
use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Contracts\TemplateLocatorInterface;
use boru\dweb\Http\HttpException;
use boru\dweb\Rendering\DWebSmarty;
use boru\dweb\Routing\CanonicalUrl;
use boru\dweb\Contracts\LayoutResolverInterface;

class SmartyRenderer implements RendererInterface
{
    /** @var \DWebSmarty */
    private $smarty;

    /** @var TemplateLocatorInterface */
    private $locator;

    /** @var SettingsInterface */
    private $settings;

    /** @var CanonicalUrl */
    private $canonical;

    /** @var LayoutResolverInterface|null */
    private $layoutResolver;


    /** @var string */
    private $extension = 'tpl';

    public function __construct(DWebSmarty $smarty, TemplateLocatorInterface $locator, SettingsInterface $settings, CanonicalUrl $canonical, LayoutResolverInterface $layoutResolver = null)
    {
        $this->smarty   = $smarty;
        $this->locator  = $locator;
        $this->settings = $settings;
        $this->canonical = $canonical;
        $this->layoutResolver = $layoutResolver;
    }

    public function render($moduleName, $templateId, array $data = array(), $viewModuleName = null)
    {
        $path = $this->locator->resolve($moduleName, $templateId, $this->extension);
        if (!$path) {
            throw new HttpException(
                404,
                'Not Found',
                "Template not found: {$moduleName}::{$templateId}.{$this->extension}"
            );
        }

        // Clear previous assigns to avoid leakage across renders.
        $this->smarty->clearAllAssign();

        $sections = new \boru\dweb\Rendering\SectionBag();
        $assets   = new \boru\dweb\Assets\AssetManager($this->settings, $sections);

        // Provide helper as $dweb (explicit, override-aware)
        $basePath = '';
        if (isset($_SERVER['SCRIPT_NAME'])) {
            $dir = rtrim(str_replace('\\', '/', dirname((string)$_SERVER['SCRIPT_NAME'])), '/');
            if ($dir === '.' || $dir === '\\' || $dir === '/') $dir = '';
            $basePath = $dir;
        }
        // Use viewModuleName (controller module) if provided; otherwise fall back
        // to the module used for template resolution (backwards compatible).
        $helperModule = $viewModuleName !== null ? (string)$viewModuleName : (string)$moduleName;

        $this->smarty->assign('dweb', new TemplateHelper(
            $this->settings,
            $this->locator,
            $helperModule,
            $basePath,
            $this->canonical,
            $sections,
            $assets
        ));

        foreach ($data as $k => $v) {
            $this->smarty->assign((string)$k, $v);
        }

        // Render by absolute path
        return $this->smarty->fetch('file:' . $path);
    }

    public function renderLayout($moduleName, $layoutId, $templateId, array $data = array())
    {
        // Render inner view with controller's module name (unchanged)
        $content = $this->render($moduleName, $templateId, $data);

        $layoutData = $data;
        $layoutData['content'] = $content;

        // Resolve layout module + id from selector
        $resolved        = $this->resolveLayout($moduleName, $layoutId);
        $layoutModule    = $resolved['module'];
        $resolvedLayoutId = $resolved['layout'];

        // For the layout render:
        // - Use $layoutModule to find the layout template
        // - But keep $dweb->module() as the original controller module ($moduleName)
        return $this->render($layoutModule, $resolvedLayoutId, $layoutData, $moduleName);
    }



    /**
     * Helper: build a configured Smarty instance based on dweb settings.
     */
    public static function buildSmarty(SettingsInterface $settings)
    {
        $compileDir = (string)$settings->get('dweb.smarty.compile_dir', sys_get_temp_dir() . '/dweb_smarty/compile');
        $cacheDir   = (string)$settings->get('dweb.smarty.cache_dir',   sys_get_temp_dir() . '/dweb_smarty/cache');

        self::ensureDir($compileDir);
        self::ensureDir($cacheDir);

        $smarty = new DWebSmarty();
        $smarty->setCompileDir($compileDir);
        $smarty->setCacheDir($cacheDir);

        // sane defaults
        $smarty->setCompileCheck((bool)$settings->get('dweb.smarty.compile_check', true));
        $smarty->setCaching((bool)$settings->get('dweb.smarty.caching', false));
        $smarty->setCacheLifetime((int)$settings->get('dweb.smarty.cache_lifetime', 0));

        // Security note: no template_dir needed since we render via file: absolute path
        return $smarty;
    }

    private static function ensureDir($dir)
    {
        $dir = rtrim((string)$dir, DIRECTORY_SEPARATOR);
        if ($dir === '') return;

        if (is_dir($dir)) return;

        if (!@mkdir($dir, 0775, true) && !is_dir($dir)) {
            throw new HttpException(
                503,
                'Service Unavailable',
                'Smarty directories could not be initialized.',
                "Failed to create directory: {$dir}"
            );
        }
    }

    /**
     * Render a layout given fully prepared section data.
     *
     * This mirrors renderLayout() but assumes the caller has already
     * prepared "content" and "sections" in $layoutData.
     *
     * @param string $moduleName
     * @param string $layoutId
     * @param array  $layoutData
     * @return string
     */
    public function renderLayoutWithSections($moduleName, $layoutId, array $layoutData = array())
    {
        // Resolve layout module + id from selector (same as renderLayout)
        $resolved         = $this->resolveLayout($moduleName, $layoutId);
        $layoutModule     = $resolved['module'];
        $resolvedLayoutId = $resolved['layout'];

        // For the layout render:
        // - Use $layoutModule to find the layout template
        // - But keep $dweb->module() as the original controller module ($moduleName)
        return $this->render($layoutModule, $resolvedLayoutId, $layoutData, $moduleName);
    }

    /**
     * Resolve layout module/id based on controller module + selector string.
     *
     * @param string $controllerModule
     * @param string $layoutId
     * @return array ['module' => string, 'layout' => string]
     */
    private function resolveLayout($controllerModule, $layoutId)
    {
        if ($this->layoutResolver === null) {
            // No resolver wired; preserve old behavior for layout selection.
            return array(
                'module' => (string)$controllerModule,
                'layout' => (string)$layoutId,
            );
        }

        return $this->layoutResolver->resolve($controllerModule, $layoutId);
    }

    /**
     * Allow "Module:template" syntax for template/layout ids when resolving.
     *
     * If no module prefix is given, falls back to the provided $fallbackModule.
     *
     * @param string $fallbackModule
     * @param string $id
     * @return array [string $module, string $tpl]
     */
    private function parseModuleTemplate($fallbackModule, $id)
    {
        $fallbackModule = (string)$fallbackModule;
        $id = (string)$id;

        $pos = strpos($id, ':');
        if ($pos === false) {
            return array($fallbackModule, $id);
        }

        $module = substr($id, 0, $pos);
        $tpl    = substr($id, $pos + 1);

        if ($module === '') {
            $module = $fallbackModule;
        }

        return array($module, $tpl);
    }

}
