<?php
namespace boru\dweb\Rendering;

use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Contracts\TemplateLocatorInterface;
use boru\dweb\Routing\CanonicalUrl;

/**
 * Template helper exposed to templates as $dweb.
 *
 * Purpose:
 * - Resolve override-aware template paths explicitly (no magic inference).
 * - Provide Smarty-friendly include strings ("file:/abs/path.tpl").
 * - Provide PHP-friendly absolute paths for require/include.
 */
class TemplateHelper
{
    /** @var TemplateLocatorInterface */
    private $locator;

    /** @var string */
    private $moduleName;

    /** @var string */
    private $smartyExt = 'tpl';

    /** @var string */
    private $phpExt = 'php';

    /** @var string */
    private $basePath;

    /** @var CanonicalUrl */
    private $canonical;

    /** @var SettingsInterface */
    private $settings;

    private $sections;
    private $assets;

    public function __construct(SettingsInterface $settings, TemplateLocatorInterface $locator, $moduleName, $basePath, CanonicalUrl $canonical, $sections, $assets)
    {
        $this->settings   = $settings;
        $this->locator    = $locator;
        $this->moduleName = (string)$moduleName;
        $this->basePath   = (string)$basePath;
        $this->canonical  = $canonical;
        $this->sections   = $sections;
        $this->assets     = $assets;
    }

    public function basePath()
    {
        return $this->basePath;
    }

    /**
     * Build a public URL to a published asset.
     *
     * Usage:
     *   {$dweb->asset('dweb/dweb.js')}
     *   {$dweb->asset('Skeleton/skeleton.css')}
     *
     * @param string $path
     * @return string
     */
    public function asset($path)
    {
        return $this->assetUrl($path);
    }

    /**
     * Build a public URL to a published asset.
     *
     * @param string $path
     * @return string
     */
    public function assetUrl($path)
    {
        $path = ltrim((string)$path, '/');

        $prefix = (string)$this->settings->get(
            \boru\dweb\Config\ConfigKeys::ASSET_URL_PREFIX,
            '/assets'
        );

        $prefix = rtrim($prefix, '/');

        return $prefix . '/' . $path;
    }


    public function url($path)
    {
        $path = (string)$path;
        if ($path === '') return $this->basePath !== '' ? $this->basePath . '/' : '/';
        if ($path[0] !== '/') $path = '/' . $path;
        return $this->basePath !== '' ? ($this->basePath . $path) : $path;
    }

    /**
     * Resolve a template id to an absolute path (override-aware).
     *
     * @param string $templateId e.g. "partials/hello"
     * @param string|null $moduleName optional different module
     * @param string $extension "tpl" or "php"
     * @return string|null
     */
    public function path($templateId, $moduleName = null, $extension = 'tpl')
    {
        $templateId = (string)$templateId;
        $extension  = (string)$extension;

        // Start from explicit module if provided, otherwise the helper's module
        $module = $moduleName !== null ? (string)$moduleName : $this->moduleName;

        // Allow "Module:template" syntax in $templateId
        $pos = strpos($templateId, ':');
        if ($pos !== false) {
            $prefix = substr($templateId, 0, $pos);
            $tpl    = substr($templateId, $pos + 1);

            if ($prefix !== '') {
                $module = $prefix;
            }

            $templateId = $tpl;
        }

        return $this->locator->resolve($module, $templateId, $extension);
    }

    /**
     * Smarty include string ("file:/abs/path.tpl") or null if missing.
     *
     * Usage in Smarty:
     *   {include file=$dweb->file('partials/hello')}
     *
     * @param string $templateId
     * @param string|null $moduleName
     * @return string|null
     */
    public function file($templateId, $moduleName = null)
    {
        $path = $this->path($templateId, $moduleName, $this->smartyExt);
        return $path ? ('file:' . $path) : null;
    }

    /**
     * Convenience: resolve "partials/<id>"
     *
     * @param string $partialId e.g. "hello"
     * @param string|null $moduleName
     * @return string|null
     */
    public function partialFile($partialId, $moduleName = null)
    {
        return $this->file('partials/' . (string)$partialId, $moduleName);
    }

    /**
     * Convenience: resolve "fragments/<id>"
     *
     * @param string $fragmentId e.g. "hello"
     * @param string|null $moduleName
     * @return string|null
     */
    public function fragmentFile($fragmentId, $moduleName = null)
    {
        return $this->file('fragments/' . (string)$fragmentId, $moduleName);
    }

    /**
     * PHP-template convenience for require/include.
     *
     * Usage in PHP templates:
     *   $p = $dweb->phpPath('partials/hello');
     *   if ($p) require $p;
     *
     * @param string $templateId
     * @param string|null $moduleName
     * @return string|null
     */
    public function phpPath($templateId, $moduleName = null)
    {
        return $this->path($templateId, $moduleName, $this->phpExt);
    }

    public function viewUrl($qualified /*, $tail... */)
    {
        $args = func_get_args();
        $qualified = isset($args[0]) ? $args[0] : '';
        $tail = $this->normalizeTailArgs($args, 1);
        return $this->canonical->view($qualified, $tail);
    }

    public function actionUrl($qualified /*, $tail... */)
    {
        $args = func_get_args();
        $qualified = isset($args[0]) ? $args[0] : '';
        $tail = $this->normalizeTailArgs($args, 1);
        return $this->canonical->action($qualified, $tail);
    }

    /**
     * Module-local view sugar:
     *   {$dweb->view('home')}
     *   {$dweb->view('home', ['13','something'])}
     *   {$dweb->view('home', '13/something')}
     *   {$dweb->view('home', '13', 'something', 'else')}
     */
    public function view($viewName /*, $tail... */)
    {
        $args = func_get_args();
        $viewName = isset($args[0]) ? $args[0] : '';
        $tail = $this->normalizeTailArgs($args, 1);

        $qualified = $this->canonical->qualifyView($this->moduleName, (string)$viewName);
        return $this->canonical->view($qualified, $tail);
    }

    public function viewWithParams($name, array $params = array())
    {
        $url = $this->view($name);
        if (!$params) return $url;

        $qs = http_build_query($params);
        return (strpos($url, '?') !== false)
            ? $url . '&' . $qs
            : $url . '?' . $qs;
    }

    /**
     * Module-local action sugar:
     *   {$dweb->action('ping')}
     *   {$dweb->action('ping', ['13'])}
     *   {$dweb->action('ping', '13')}
     *   {$dweb->action('ping', '13', 'something')}
     */
    public function action($actionName /*, $tail... */)
    {
        $args = func_get_args();
        $actionName = isset($args[0]) ? $args[0] : '';
        $tail = $this->normalizeTailArgs($args, 1);

        $qualified = $this->canonical->qualifyAction($this->moduleName, (string)$actionName);
        return $this->canonical->action($qualified, $tail);
    }

    public function actionWithParams($name, array $params = array())
    {
        $url = $this->action($name);
        if (!$params) return $url;

        $qs = http_build_query($params);
        return (strpos($url, '?') !== false)
            ? $url . '&' . $qs
            : $url . '?' . $qs;
    }

    /**
     * Cross-module sugar:
     *   {$dweb->mView('Other', 'home')}
     *   {$dweb->mView('Other', 'home', '13', 'something')}
     *   {$dweb->mAction('Other', 'echo', ['13'])}
     */
    public function mView($moduleName, $viewName /*, $tail... */)
    {
        $args = func_get_args();
        $moduleName = isset($args[0]) ? $args[0] : '';
        $viewName   = isset($args[1]) ? $args[1] : '';
        $tail = $this->normalizeTailArgs($args, 2);

        $qualified = $this->canonical->qualifyView((string)$moduleName, (string)$viewName);
        return $this->canonical->view($qualified, $tail);
    }

    public function mAction($moduleName, $actionName /*, $tail... */)
    {
        $args = func_get_args();
        $moduleName = isset($args[0]) ? $args[0] : '';
        $actionName = isset($args[1]) ? $args[1] : '';
        $tail = $this->normalizeTailArgs($args, 2);

        $qualified = $this->canonical->qualifyAction((string)$moduleName, (string)$actionName);
        return $this->canonical->action($qualified, $tail);
    }

    public function canonicalEnabled()
    {
        return $this->canonical->enabled();
    }

    public function module()
    {
        return $this->moduleName;
    }

    public function assets()
    {
        return $this->assets;
    }

    public function section($name, $default = '')
    {
        if ($this->sections === null) return (string)$default;

        $html = $this->sections->get((string)$name, (string)$default);
        return $html === '' ? '' : ($html . "\n");
    }

    /**
     * Convention selector for a section name: "#section-<name>".
     * Useful in templates for hx-target, anchors, etc.
     *
     * @param string $sectionName
     * @return string
     */
    public function sectionSelector($sectionName)
    {
        return '#section-' . (string)$sectionName;
    }

    /**
     * Convenience: stable DOM id for a section ("section-<name>").
     *
     * @param string $sectionName
     * @return string
     */
    public function sectionId($sectionName)
    {
        return 'section-' . (string)$sectionName;
    }

    /**
     * Normalize "tail" arguments from Smarty-friendly calls.
     *
     * Supported:
     *  - view('home') => null
     *  - view('home', ['13','something'])
     *  - view('home', '13/something')
     *  - view('home', '13', 'something', 'else')  (varargs)
     *
     * @param array $args full func_get_args() array
     * @param int $start index where tail begins
     * @return array|string|null
     */
    private function normalizeTailArgs(array $args, $start)
    {
        $start = (int)$start;
        if (count($args) <= $start) return null;

        // If exactly one tail argument is provided, preserve it:
        // - array => array tail
        // - string => string tail (CanonicalUrl supports "a/b/c" or single segment)
        if (count($args) === ($start + 1)) {
            $t = $args[$start];
            if ($t === null) return null;
            return $t;
        }

        // If multiple args are provided, treat them as tail segments (array)
        $out = array();
        for ($i = $start; $i < count($args); $i++) {
            $v = $args[$i];
            if ($v === null) continue;
            $s = trim((string)$v);
            if ($s === '') continue;
            $out[] = $s;
        }

        return count($out) ? $out : null;
    }

    // In boru\dweb\Rendering\TemplateHelper (or wherever $dweb lives)

    /**
     * Emit a marker that tells dweb.js to stop HTMX SSE reconnect.
     *
     * @param string|null $freezeSelector e.g. "#sse-static" to copy status/stream into a static target
     * @param bool $removeRoot if true, removes the hx-ext="sse" root node to hard-stop reconnect
     * @return string HTML
     */
    public function sseDoneMarker($freezeSelector = null, $removeRoot = true)
    {
        $attrs = $this->sseDoneAttrs($freezeSelector, $removeRoot);
        return '<span ' . $attrs . '></span>';
    }

    /**
     * Marker attributes (for embedding into your own element).
     *
     * @param string|null $freezeSelector
     * @param bool $removeRoot
     * @return string
     */
    public function sseDoneAttrs($freezeSelector = null, $removeRoot = true)
    {
        $out = array();
        $out[] = 'data-dweb-sse-done="1"';

        if ($freezeSelector) {
            $out[] = 'data-dweb-sse-freeze="' . htmlspecialchars((string)$freezeSelector, ENT_QUOTES, 'UTF-8') . '"';
        }
        if ($removeRoot) {
            $out[] = 'data-dweb-sse-remove="1"';
        }

        return implode(' ', $out);
    }

    /**
     * Returns socket-related settings as an array.
     * @return array 
     */
    public function socketSettings() {
        $socketUrl = $this->settings->get(\boru\dweb\Config\ConfigKeys::SOCKET_PUBLIC_URL, null);
        if($socketUrl === null) {
            $socketUrl = 'http://127.0.0.1:' . $this->settings->get(\boru\dweb\Config\ConfigKeys::SOCKET_NODE_PORT, 3001);
        }
        return [
            "enabled"=>$this->settings->get(\boru\dweb\Config\ConfigKeys::SOCKET_ENABLED, false),
            "authMode"=>$this->settings->get(\boru\dweb\Config\ConfigKeys::SOCKET_AUTH_MODE, 'jwt'),
            "jwtSecret"=>$this->settings->get(\boru\dweb\Config\ConfigKeys::SOCKET_JWT_SECRET, null),
            "tokenTtl"=>$this->settings->get(\boru\dweb\Config\ConfigKeys::SOCKET_TOKEN_TTL, 3600),
            "tokenUrl"=>$this->settings->get(\boru\dweb\Config\ConfigKeys::SOCKET_TOKEN_URL, null),
            "socketUrl"=>$socketUrl,
        ];
    }
}
