<?php
namespace boru\dweb\Routing;

/**
 * Thin facade that prefixes route keys with "<Module>."
 * while keeping module code unaware of global naming.
 */
class ModuleRouteCollection
{
    /** @var RouteCollection */
    private $routes;

    /** @var string */
    private $moduleName;

    public function __construct(RouteCollection $routes, $moduleName)
    {
        $this->routes     = $routes;
        $this->moduleName = (string)$moduleName;
    }

    public function addView($name, $handler, $pathParams = null, $pathPattern = null)
    {
        return $this->routes->addView(
            $this->qualify($name),
            $handler,
            $pathParams,
            $pathPattern
        );
    }

    public function addAction($name, $handler, $methods = null, $pathParams = null, $pathPattern = null)
    {
        return $this->routes->addAction(
            $this->qualify($name),
            $handler,
            $methods,
            $pathParams,
            $pathPattern
        );
    }

    public function qualify($name)
    {
        $name = (string)$name;
        if ($name === '') return $this->moduleName; // optional: allow module root
        return $this->moduleName . '.' . $name;
    }

    /**
     * Register a module view by name using the default factory wiring.
     *
     * @param string $name
     * @param \boru\dweb\Kernel\Container $container
     * @param array|null $pathParams
     * @param string|null $pathPattern
     * @return void
     */
    public function addModuleView($name, $container, $pathParams = null, $pathPattern = null)
    {
        $name = (string)$name;
        return $this->addView(
            $name,
            new ModuleViewHandler($this->moduleName, $name, $container),
            $pathParams,
            $pathPattern
        );
    }

    /**
     * Register a module action by name using the default factory wiring.
     *
     * @param string $name
     * @param \boru\dweb\Kernel\Container $container
     * @param array|string|null $methods
     * @param array|null $pathParams
     * @param string|null $pathPattern
     * @return void
     */
    public function addModuleAction($name, $container, $methods = null, $pathParams = null, $pathPattern = null)
    {
        $name = (string)$name;
        return $this->addAction(
            $name,
            new ModuleActionHandler($this->moduleName, $name, $container),
            $methods,
            $pathParams,
            $pathPattern
        );
    }
}
