<?php
namespace boru\dweb\Routing;

use boru\dweb\Config\ConfigKeys;
use boru\dweb\Contracts\SettingsInterface;

class RouteCollection
{
    /** @var array<string, array> */
    private $actions = array();

    /** @var array<string, array> */
    private $views = array();

    /**
     * Optional explicit path patterns for views/actions.
     * key: qualified name ("Module.view" or "Module.action")
     * val: string pattern like "/Module/foo/{id}"
     *
     * NOTE: we only use patterns for views in PathRouter today, but
     *       actions are supported here for future use.
     *
     * @var array<string,string>
     */
    private $pathPatterns = array();

    /**
     * @param string $name qualified name: "Module.action"
     * @param callable $handler
     * @param array|string|null $methods
     * @param array|null $pathParams ordered param names: ['id','slug']
     * @param string|null $pathPattern optional explicit path pattern
     * @return $this
     */
    public function addAction($name, $handler, $methods = null, $pathParams = null, $pathPattern = null)
    {
        $name = (string)$name;

        $m = null;
        if ($methods !== null) {
            if (!is_array($methods)) $methods = array($methods);
            $m = array();
            foreach ($methods as $mm) {
                $mm = strtoupper((string)$mm);
                if ($mm !== '') $m[$mm] = true;
            }
        }

        $pp = null;
        if (is_array($pathParams)) {
            $pp = array();
            foreach ($pathParams as $p) {
                $p = trim((string)$p);
                if ($p !== '') $pp[] = $p;
            }
            if (count($pp) === 0) $pp = null;
        }

        $this->actions[$name] = array(
            'handler'    => $handler,
            'methods'    => $m,   // null means "any"
            'pathParams' => $pp,  // null means "no named params"
        );

        $pathPattern = (string)$pathPattern;
        if ($pathPattern !== '') {
            $this->pathPatterns[$name] = $pathPattern;
        }

        return $this;
    }

    /**
     * @param string $name qualified name: "Module.view"
     * @param callable $handler
     * @param array|null $pathParams ordered param names: ['id','slug']
     * @param string|null $pathPattern optional explicit path pattern
     * @return $this
     */
    public function addView($name, $handler, $pathParams = null, $pathPattern = null)
    {
        $name = (string)$name;

        $pp = null;
        if (is_array($pathParams)) {
            $pp = array();
            foreach ($pathParams as $p) {
                $p = trim((string)$p);
                if ($p !== '') $pp[] = $p;
            }
            if (count($pp) === 0) $pp = null;
        }

        $this->views[$name] = array(
            'handler'    => $handler,
            'pathParams' => $pp,
        );

        $pathPattern = (string)$pathPattern;
        if ($pathPattern !== '') {
            $this->pathPatterns[$name] = $pathPattern;
        }

        return $this;
    }

    public function getAction($name)
    {
        $name = (string)$name;
        return isset($this->actions[$name]) ? $this->actions[$name] : null;
    }

    /**
     * Back-compat: return callable handler (old behavior)
     */
    public function getView($name)
    {
        $rec = $this->getViewRecord($name);
        return $rec ? $rec['handler'] : null;
    }

    public function getViewRecord($name)
    {
        $key = (string)$name;
        return isset($this->views[$key]) ? $this->views[$key] : null;
    }

    public function forModule($moduleName)
    {
        return new ModuleRouteCollection($this, $moduleName);
    }

    /**
     * Back-compat: return name => callable handlers (used by RouteDump)
     * @return array<string, callable>
     */
    public function allViews()
    {
        $out = array();
        foreach ($this->views as $name => $rec) {
            $out[$name] = $rec['handler'];
        }
        return $out;
    }

    /**
     * name => ['handler'=>callable,'methods'=>array|null,'pathParams'=>array|null]
     */
    public function allActions()
    {
        return $this->actions;
    }

    /**
     * Find a view route for the given module and full path segments.
     *
     * Tries explicit path patterns first, then falls back to legacy
     *   /<Module>/<View>/<tail...>
     * behavior.
     *
     * @param string $moduleName
     * @param array  $parts   full path segments (without leading slash)
     * @param SettingsInterface $config
     * @return array|null ['qualified'=>string,'record'=>array,'tail'=>array]
     */
    public function matchModulePath($moduleName, array $parts, $config)
    {
        $moduleName = (string)$moduleName;

        // 1) Try explicit patterns first
        foreach ($this->views as $qualified => $rec) {
            // qualified name is "Module.view"
            if (strpos($qualified, $moduleName . '.') !== 0) {
                continue;
            }

            if (!isset($this->pathPatterns[$qualified])) {
                continue;
            }

            $pattern = $this->pathPatterns[$qualified];
            $match   = $this->matchPathPattern($pattern, $parts);
            if ($match !== null) {
                return array(
                    'qualified' => $qualified,
                    'record'    => $rec,
                    'tail'      => $match['tail'],
                );
            }
        }

        // 2) Fallback to legacy /<Module>/<View>/<tail...>
        $defaultView = $config->get(ConfigKeys::DEFAULT_VIEW, '');

        $viewName = isset($parts[1]) && $parts[1] !== '' ? $parts[1] : $defaultView;
        $tail     = array_slice($parts, 2);

        $qualified = $moduleName . '.' . $viewName;
        $rec       = $this->getViewRecord($qualified);
        if ($rec) {
            return array(
                'qualified' => $qualified,
                'record'    => $rec,
                'tail'      => $tail,
            );
        }

        return null;
    }

    /**
     * Very small pattern matcher.
     *
     * Example pattern: "/Module/foo/{id}/{sub}"
     *
     * - Literal segments must match exactly.
     * - Segments of the form "{name}" accept and capture any non-empty value.
     * - The number of segments must match exactly.
     *
     * Returns ['tail' => array] where tail values are ordered to align with
     * the route's pathParams schema in ModuleRouteCollection.
     *
     * @param string $pattern
     * @param array  $parts
     * @return array|null
     */
    private function matchPathPattern($pattern, array $parts)
    {
        $pattern = trim((string)$pattern);
        if ($pattern === '') return null;

        $patternParts = explode('/', trim($pattern, '/'));
        $pathParts    = $parts;

        if (count($patternParts) !== count($pathParts)) {
            return null;
        }

        $tail = array();

        foreach ($patternParts as $i => $seg) {
            $pathSeg = isset($pathParts[$i]) ? $pathParts[$i] : null;
            if ($pathSeg === null || $pathSeg === '') {
                return null;
            }

            $len = strlen($seg);
            if ($len > 2 && $seg[0] === '{' && $seg[$len - 1] === '}') {
                // Named parameter, capture value in order
                $tail[] = $pathSeg;
                continue;
            }

            // Literal segment
            if ($seg !== $pathSeg) {
                return null;
            }
        }

        return array('tail' => $tail);
    }
}
