<?php
namespace boru\dweb\Routing;

use boru\dweb\Http\Request;
use boru\dweb\Config\ConfigKeys;
use boru\dweb\Contracts\SettingsInterface;

class WebRouter
{
    /** @var SettingsInterface */
    private $config;

    /** @var RouteCollection */
    private $routes;

    public function __construct(SettingsInterface $config, RouteCollection $routes)
    {
        $this->config = $config;
        $this->routes = $routes;
    }

    public function dispatch(Request $req)
    {
        $tail = $this->readTailFromQuery($req);

        // ACTIONS
        $actionParam = $this->config->get(ConfigKeys::ACTION_PARAM, 'action');
        $action = $req->param($actionParam, null);
        if ($action) {
            $action = $this->qualifyIfNeeded($action, $this->config->get(ConfigKeys::DEFAULT_MODULE, null));

            $record = $this->routes->getAction($action);
            if ($record && $this->actionAllowsMethod($record, $req->method())) {
                $schema = isset($record['pathParams']) ? $record['pathParams'] : null;
                $req2 = $this->attachPathParams($req, $action, $tail, $schema);
                return call_user_func($record['handler'], $req2);
            }

            throw new \boru\dweb\Http\HttpException(
                $record ? 405 : 404,
                $record ? 'Method Not Allowed' : 'Not Found',
                ($record ? 'Method not allowed for action: ' : 'Action not found: ') . $action
            );
        }

        // VIEWS
        $pageParam = $this->config->get(ConfigKeys::PAGE_PARAM, 'view');
        $defaultView = $this->config->get(ConfigKeys::DEFAULT_VIEW, '');
        $view = $req->param($pageParam, $defaultView);
        $view = $this->qualifyIfNeeded($view, $this->config->get(ConfigKeys::DEFAULT_MODULE, null));

        // Use record so we can get schema/pathParams too
        $rec = method_exists($this->routes, 'getViewRecord')
            ? $this->routes->getViewRecord($view)
            : null;

        if ($rec && isset($rec['handler'])) {
            $schema = isset($rec['pathParams']) ? $rec['pathParams'] : null;
            $req2 = $this->attachPathParams($req, $view, $tail, $schema);
            return call_user_func($rec['handler'], $req2);
        }

        // back-compat fallback if getViewRecord doesn't exist
        $handler = $this->routes->getView($view);
        if ($handler) {
            $req2 = $this->attachPathParams($req, $view, $tail, null);
            return call_user_func($handler, $req2);
        }

        // fallback to "" (root) if a module registered it as default
        $fallback = $this->routes->getView('');
        if ($fallback) {
            $req2 = $this->attachPathParams($req, '', $tail, null);
            return call_user_func($fallback, $req2);
        }

        throw new \boru\dweb\Http\NotFoundException('View not found: ' . $view);
    }

    private function qualifyIfNeeded($id, $defaultModule)
    {
        $id = (string)$id;
        if ($id === '') return $id;

        if (strpos($id, '.') !== false) return $id;
        if (!$defaultModule) return $id;

        return $defaultModule . '.' . $id;
    }

    private function actionAllowsMethod($actionRecord, $method)
    {
        if (!$actionRecord) return false;
        if (!isset($actionRecord['methods']) || $actionRecord['methods'] === null) return true;
        return isset($actionRecord['methods'][strtoupper((string)$method)]);
    }

    /**
     * Query-mode tail support:
     *   ?view=Skeleton.home&tail[]=13&tail[]=something
     *   ?action=Skeleton.ping&tail[]=13
     *
     * Also accepts tail as a single string for convenience:
     *   &tail=13/something
     */
    private function readTailFromQuery(Request $req)
    {
        $t = $req->param('tail', null);
        if ($t === null || $t === false) return array();

        if (is_array($t)) {
            $out = array();
            foreach ($t as $v) {
                if ($v === null) continue;
                $s = trim((string)$v);
                if ($s === '') continue;
                $out[] = $s;
            }
            return $out;
        }

        $s = trim((string)$t);
        if ($s === '') return array();
        if (strpos($s, '/') !== false) {
            $parts = explode('/', $s);
            $out = array();
            foreach ($parts as $p) {
                $p = trim((string)$p);
                if ($p !== '') $out[] = $p;
            }
            return $out;
        }

        return array($s);
    }

    private function attachPathParams(Request $req, $qualified, array $tail, $schema)
    {
        $named = array();
        if (is_array($schema)) {
            $i = 0;
            foreach ($schema as $key) {
                if (!isset($tail[$i])) break;
                $named[(string)$key] = $tail[$i];
                $i++;
            }
        }

        // Requires Request::withMeta(), which you said you added.
        return $req
            ->withMeta('dweb.route', (string)$qualified)
            ->withMeta('dweb.path.tail', $tail)
            ->withMeta('dweb.path.params', $named);
    }
}
