<?php
namespace boru\dweb\Support;

use boru\dweb\DwebConfig;
use boru\dweb\WebUI;
use boru\dweb\Config\ConfigKeys;
use boru\dweb\Modules\Core\CoreModule;
use boru\dweb\Kernel\ModuleManager;

/**
 * Shared boot entry for:
 * - host app public/index.php
 * - tests
 * - vendor/bin scripts
 *
 * Uses KernelBuilder as single wiring source of truth.
 */
final class Bootstrap
{
    private function __construct() {}

    /**
     * @return string absolute package root (vendor/bin safe)
     */
    public static function packageRoot()
    {
        $rc = new \ReflectionClass(__CLASS__);
        $file = $rc->getFileName(); // .../src/Support/Bootstrap.php
        return dirname(dirname(dirname($file)));
    }

    /**
     * Build a default WebConfig + apply overrides.
     *
     * @param array $opts
     * @param callable|null $configure function(WebConfig $config): void
     * @return DwebConfig
     */
    public static function config(array $opts = array(), $configure = null)
    {
        $config = DwebConfig::createDefault();

        // keep your deterministic defaults style, but allow caller to override
        if (!isset($opts[ConfigKeys::DEBUG_ENABLED]))        $config->set(ConfigKeys::DEBUG_ENABLED, true);
        if (!isset($opts[ConfigKeys::DEBUG_ROUTES_ENABLED])) $config->set(ConfigKeys::DEBUG_ROUTES_ENABLED, true);

        // Deterministic defaults (copied from your build_env)
        $config->set(ConfigKeys::DEFAULT_MODULE, isset($opts['defaultModule']) ? $opts['defaultModule'] : 'Skeleton');
        $config->set(ConfigKeys::DEFAULT_VIEW, isset($opts['defaultPage']) ? $opts['defaultPage'] : 'home');
        $config->set(ConfigKeys::PAGE_PARAM, isset($opts['pageParam']) ? $opts['pageParam'] : 'view');
        $config->set(ConfigKeys::ACTION_PARAM, isset($opts['actionParam']) ? $opts['actionParam'] : 'action');

        // Security headers ON by default (copied)
        $config->set(ConfigKeys::SECURITY_HEADERS_ENABLED, true);
        $config->set(ConfigKeys::SECURITY_HEADERS_CONTENT_SECURITY_POLICY, null);
        $config->set(ConfigKeys::SECURITY_HEADERS_X_FRAME_OPTIONS, null);
        $config->set(ConfigKeys::SECURITY_HEADERS_X_CONTENT_TYPE_OPTIONS, null);
        $config->set(ConfigKeys::SECURITY_HEADERS_X_XSS_PROTECTION, null);

        // Apply key=>value overrides (ConfigKeys etc.)
        foreach ($opts as $k => $v) {
            // skip the special non-config options above
            if ($k === 'defaultModule' || $k === 'defaultPage' || $k === 'pageParam' || $k === 'actionParam') {
                continue;
            }
            $config->set((string)$k, $v);
        }

        if ($configure) {
            call_user_func($configure, $config);
        }

        return $config;
    }

    /**
     * Web runner for host apps.
     *
     * NOTE: WebUI likely already has logic to build container/router internally.
     * The recommended direction is to refactor WebUI to call KernelBuilder::build()
     * so WebUI, tests, and vendor/bin share identical wiring.
     *
     * @param array $opts
     * @param callable|null $configure function(WebConfig $config, WebUI $ui): void
     * @return void
     */
    public static function run(array $opts = array(), $configure = null)
    {
        $config = self::config($opts);

        $ui = new WebUI($config);

        if ($configure) {
            call_user_func($configure, $config, $ui);
        }

        $ui->render();
    }

    /**
     * Build a KernelEnv from a shared host env file that returns:
     *
     *   [
     *     'config'         => DwebConfig,
     *     'modules'        => ModuleInterface[],
     *     'container'      => callable|null,
     *     'web_middleware' => MiddlewareInterface[]|null,
     *   ]
     *
     * CoreModule is *always* injected here, before host modules.
     *
     * @param string     $path
     * @param array|null $middlewareOverride If non-null, overrides web_middleware from file.
     * @return \boru\dweb\Support\KernelEnv
     */
    public static function kernelEnvFromHostFile($path, array $middlewareOverride = null)
    {
        if (!file_exists($path)) {
            throw new \RuntimeException("Env file not found: " . $path);
        }

        $data = require $path;

        if (!is_array($data) || !isset($data['config'])) {
            throw new \RuntimeException("Env file must return an array with at least a 'config' key: " . $path);
        }

        /** @var \boru\dweb\DwebConfig $config */
        $config = $data['config'];

        $hostModules = isset($data['modules']) && is_array($data['modules'])
            ? $data['modules']
            : array();

        $containerCb = isset($data['container'])
            ? $data['container']
            : null;

        $defaultMw = isset($data['web_middleware']) && is_array($data['web_middleware'])
            ? $data['web_middleware']
            : null;

        $middleware = $middlewareOverride !== null ? $middlewareOverride : $defaultMw;

        $spec = \boru\dweb\Support\KernelSpec::fromConfig($config)
            ->withContainer($containerCb)
            ->withModules(function (ModuleManager $mm, $c, $cfg) use ($hostModules) {
                // Framework core
                $mm->add(new CoreModule());

                // Host modules
                foreach ($hostModules as $m) {
                    $mm->add($m);
                }
            })
            ->withMiddleware($middleware);

        return \boru\dweb\Support\KernelBuilder::buildEnv($spec);
    }

}
