<?php
namespace boru\dweb\Support;

use boru\dweb\Contracts\ControllerContextInterface;
use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Contracts\CliInvokerInterface;
use boru\dweb\Contracts\LoggerInterface;
use boru\dweb\Mvc\DwebHelper;
use boru\dweb\Assets\AssetManager;
use boru\dweb\Support\SocketTokenService;
use boru\dweb\Support\SocketPublisher;

class ControllerContext implements ControllerContextInterface
{
    /** @var SettingsInterface */
    private $settings;

    /** @var DwebHelper */
    private $dweb;

    /** @var CliInvokerInterface|null */
    private $cli;

    /** @var LoggerInterface|null */
    private $logger;

    /** @var AssetManager|null */
    private $assets;

    /** @var SocketTokenService|null */
    private $socketTokenService;

    /** @var SocketPublisher|null */
    private $socketPublisher;

    /** @var object|null */
    private $userIdentity = null;

    /** @var \boru\dweb\Http\Request|null */
    private $request;

    /** @var \boru\dweb\Routing\RouteCollection|null */
    private $routes;

    public function __construct(SettingsInterface $settings, DwebHelper $dweb)
    {
        $this->settings = $settings;
        $this->dweb = $dweb;
        $this->cli = null;
        $this->logger = null;
        $this->assets = null;
        $this->socketTokenService = null;
        $this->socketPublisher = null;
        $this->userIdentity = null;
        $this->request = null;
        $this->routes  = null;
    }

    /**
     * Attach the current Request to this context for use in forwards.
     * @param \boru\dweb\Http\Request $request
     * @return $this
     */
    public function withRequest(\boru\dweb\Http\Request $request)
    {
        $this->request = $request;
        return $this;
    }

    /**
     * Attach the shared RouteCollection so we can resolve qualified view/action names.
     * @param \boru\dweb\Routing\RouteCollection $routes
     * @return $this
     */
    public function withRoutes(\boru\dweb\Routing\RouteCollection $routes)
    {
        $this->routes = $routes;
        return $this;
    }

    public function withCli(CliInvokerInterface $cli)
    {
        $this->cli = $cli;
        return $this;
    }

    public function withLogger(LoggerInterface $logger)
    {
        $this->logger = $logger;
        return $this;
    }

    public function withAssets(AssetManager $assets)
    {
        $this->assets = $assets;
        return $this;
    }

    public function withSocketTokenService(SocketTokenService $service)
    {
        $this->socketTokenService = $service;
        return $this;
    }

    public function withSocketPublisher(SocketPublisher $publisher)
    {
        $this->socketPublisher = $publisher;
        return $this;
    }

    public function withUserIdentity($identity)
    {
        $this->userIdentity = $identity;
        return $this;
    }

    public function dweb()
    {
        return $this->dweb;
    }

    public function cli()
    {
        return $this->cli;
    }

    public function logger()
    {
        return $this->logger;
    }

    public function settings()
    {
        return $this->settings;
    }

    public function assets()
    {
        return $this->assets;
    }

    public function socketTokenService()
    {
        return $this->socketTokenService;
    }

    public function socketPublisher()
    {
        return $this->socketPublisher;
    }

    public function userIdentity()
    {
        return $this->userIdentity;
    }

    public function session()
    {
        if ($this->request === null) {
            return null;
        }
        if (!method_exists($this->request, 'session')) {
            return null;
        }
        return $this->request->session();
    }


    public function dispatchView($qualifiedView, \boru\dweb\Http\Request $req = null)
    {
        if ($this->routes === null) {
            throw new \RuntimeException('ControllerContext::dispatchView() requires routes; call withRoutes() when constructing the context.');
        }

        $qualifiedView = (string)$qualifiedView;

        if ($req === null) {
            if ($this->request === null) {
                throw new \RuntimeException('ControllerContext::dispatchView() requires a Request; call withRequest() or pass an override.');
            }
            $req = $this->request;
        }

        $handler = $this->routes->getView($qualifiedView);
        if (!$handler) {
            throw new \RuntimeException('View not found for forward: ' . $qualifiedView);
        }

        return call_user_func($handler, $req);
    }

    public function dispatchAction($qualifiedAction, \boru\dweb\Http\Request $req = null)
    {
        if ($this->routes === null) {
            throw new \RuntimeException('ControllerContext::dispatchAction() requires routes; call withRoutes() when constructing the context.');
        }

        $qualifiedAction = (string)$qualifiedAction;

        if ($req === null) {
            if ($this->request === null) {
                throw new \RuntimeException('ControllerContext::dispatchAction() requires a Request; call withRequest() or pass an override.');
            }
            $req = $this->request;
        }

        $record = $this->routes->getAction($qualifiedAction);
        if (!$record || !isset($record['handler'])) {
            throw new \RuntimeException('Action not found for forward: ' . $qualifiedAction);
        }

        // For internal forwards we intentionally *do not* re-check HTTP method here;
        // the caller is responsible for only forwarding where it makes sense.
        $handler = $record['handler'];
        return call_user_func($handler, $req);
    }
}