<?php
namespace boru\dweb\Support;

use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Config\ConfigKeys;

/**
 * Bridge from PHP modules to the Node socket server.
 *
 * Usage:
 *   $publisher->publish('notifications', 'message', ['text' => 'hi']);
 *   $publisher->publish('user:' . $userId, 'notification', ['text' => 'hi']);
 */
class SocketPublisher
{
    /** @var SettingsInterface */
    private $settings;

    public function __construct(SettingsInterface $settings)
    {
        $this->settings = $settings;
    }

    /**
     * Publish a message to the socket server.
     *
     * @param string      $channel e.g. "notifications" or "user:123"; null/"" = broadcast
     * @param string      $event   e.g. "message" or "notification"
     * @param array|mixed $payload
     * @return bool
     */
    public function publish($channel, $event, $payload)
    {
        $baseUrl = (string)$this->settings->get(ConfigKeys::SOCKET_PUBLIC_URL, '');
        if ($baseUrl === '') {
            // default: http://127.0.0.1:{port}
            $port = (int)$this->settings->get(ConfigKeys::SOCKET_NODE_PORT, 3001);
            if ($port <= 0) $port = 3001;
            $baseUrl = 'http://127.0.0.1:' . $port;
        }

        $endpoint = rtrim($baseUrl, '/') . '/publish';

        $apiSecret = (string)$this->settings->get(ConfigKeys::SOCKET_API_SECRET, '');

        $data = array(
            'channel' => $channel !== null ? (string)$channel : null,
            'event'   => (string)$event,
            'payload' => $payload,
        );

        $json = json_encode($data);

        if (!function_exists('curl_init')) {
            // You could add a logger call here; for now just fail fast.
            return false;
        }

        $ch = curl_init($endpoint);
        curl_setopt($ch, CURLOPT_POST, true);

        $headers = array('Content-Type: application/json');
        if ($apiSecret !== '') {
            $headers[] = 'X-Dweb-Socket-Api-Secret: ' . $apiSecret;
        }

        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 2);

        $resp = curl_exec($ch);
        $err  = curl_error($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($err || $code < 200 || $code >= 300) {
            // Optional: log $err / $code / $resp using your logger
            return false;
        }

        return true;
    }
}
