<?php
namespace boru\dweb\Support;

use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Config\ConfigKeys;
use boru\dweb\Contracts\SocketOpaqueTokenStoreInterface;

class SocketTokenService
{
    const MODE_JWT    = 'jwt';
    const MODE_OPAQUE = 'opaque';

    /** @var SettingsInterface */
    private $settings;

    /** @var SocketOpaqueTokenStoreInterface|null */
    private $opaqueStore;

    /**
     * @param SettingsInterface               $settings
     * @param SocketOpaqueTokenStoreInterface|null $opaqueStore
     */
    public function __construct(SettingsInterface $settings, SocketOpaqueTokenStoreInterface $opaqueStore = null)
    {
        $this->settings   = $settings;
        $this->opaqueStore = $opaqueStore;
    }

    /**
     * Issue a token for a subject + optional extra claims.
     *
     * @param string $subject
     * @param array  $claims
     * @return string
     */
    public function issueToken($subject, array $claims = array())
    {
        $mode = (string)$this->settings->get(ConfigKeys::SOCKET_AUTH_MODE, self::MODE_JWT);

        if ($mode === self::MODE_OPAQUE) {
            return $this->issueOpaque($subject, $claims);
        }

        // default: JWT
        return $this->issueJwt($subject, $claims);
    }

    /**
     * Issue a stateless HS256 JWT.
     *
     * @param string $subject
     * @param array  $claims
     * @return string
     */
    public function issueJwt($subject, array $claims = array())
    {
        $secret = (string)$this->settings->get(ConfigKeys::SOCKET_JWT_SECRET, '');
        if ($secret === '') {
            throw new \RuntimeException(
                'Socket JWT secret not configured (' . ConfigKeys::SOCKET_JWT_SECRET . ').'
            );
        }

        $now = time();
        $ttl = (int)$this->settings->get(ConfigKeys::SOCKET_TOKEN_TTL, 3600);
        if ($ttl <= 0) {
            $ttl = 3600;
        }
        $exp = $now + $ttl;

        $payload = $claims;
        $payload['sub'] = (string)$subject;
        $payload['iat'] = $now;
        $payload['exp'] = $exp;

        return $this->encodeJwt($payload, $secret);
    }

    /**
     * Issue an opaque token backed by a store.
     *
     * @param string $subject
     * @param array  $claims
     * @return string
     */
    public function issueOpaque($subject, array $claims = array())
    {
        if ($this->opaqueStore === null) {
            throw new \RuntimeException(
                'Opaque socket tokens require a SocketOpaqueTokenStoreInterface implementation.'
            );
        }

        $ttl = (int)$this->settings->get(ConfigKeys::SOCKET_TOKEN_TTL, 3600);
        if ($ttl <= 0) {
            $ttl = 3600;
        }

        $token = $this->generateRandomToken();

        $data = array(
            'sub'        => (string)$subject,
            'claims'     => $claims,
            'expires_at' => time() + $ttl,
        );

        $this->opaqueStore->store($token, $data, $ttl);

        return $token;
    }

    /**
     * Basic HS256 JWT encoding (no external dependency).
     *
     * @param array  $payload
     * @param string $secret
     * @return string
     */
    private function encodeJwt(array $payload, $secret)
    {
        $header = array('alg' => 'HS256', 'typ' => 'JWT');

        $segments = array(
            $this->base64UrlEncode(json_encode($header)),
            $this->base64UrlEncode(json_encode($payload)),
        );

        $signingInput = implode('.', $segments);
        $signature = hash_hmac('sha256', $signingInput, (string)$secret, true);

        $segments[] = $this->base64UrlEncode($signature);

        return implode('.', $segments);
    }

    /**
     * @param string $data
     * @return string
     */
    private function base64UrlEncode($data)
    {
        $b64 = base64_encode($data);
        // URL-safe base64 (RFC 7515)
        return str_replace(array('+', '/', '='), array('-', '_', ''), $b64);
    }

    /**
     * Generate a random opaque token (hex string).
     *
     * @return string
     */
    private function generateRandomToken()
    {
        // 32 bytes -> 64 hex chars
        $bytes = null;

        if (function_exists('random_bytes')) {
            $bytes = random_bytes(32);
        } elseif (function_exists('openssl_random_pseudo_bytes')) {
            $bytes = openssl_random_pseudo_bytes(32);
        }

        if ($bytes === null) {
            // Last resort: uniqid + mt_rand (not cryptographically strong)
            $bytes = md5(uniqid(mt_rand(), true), true);
        }

        return bin2hex($bytes);
    }
}
