<?php
namespace boru\dweb;

use boru\dweb\Config\ConfigKeys;
use boru\dweb\Contracts\ModuleInterface;
use boru\dweb\Contracts\SettingsInterface;
use boru\dweb\Http\Emitter;
use boru\dweb\Kernel\Container;
use boru\dweb\Kernel\ModuleManager;
use boru\dweb\Contracts\MiddlewareInterface;
use boru\dweb\Exceptions\Fail;
use boru\dweb\Modules\Core\CoreModule;
use boru\dweb\Support\Bootstrap;
use boru\dweb\Support\KernelEnv;

class WebUI
{

    private $env;

    /** @var Container|null */
    private $container = null;

    private $containerBootstrapCallback;

    /** @var SettingsInterface */
    private $config;

    /** @var ModuleManager */
    private $moduleManager;

    /** @var MiddlewareInterface[] */
    private $middleware = [];

    /** @var callable[] */
    private $containerBootstraps = [];

    public function __construct(SettingsInterface $config)
    {
        $this->config = $config;
        $this->moduleManager = new ModuleManager();
    }

    /**
     * Build WebUI from an existing KernelEnv.
     *
     * This path assumes CoreModule and host modules are already wired in.
     *
     * @param KernelEnv $env
     * @return static
     */
    public static function fromEnv(KernelEnv $env)
    {
        $ui = new static($env->config());
        $ui->env = $env;
        return $ui;
    }

    /**
     * Host-facing convenience: build WebUI from an env.php-style file.
     *
     * @param string $path Absolute or relative path to env.php
     * @return static
     */
    public static function fromEnvFile($path)
    {
        $env = Bootstrap::kernelEnvFromHostFile($path);
        return static::fromEnv($env);
    }


    public function withModule(ModuleInterface $module)
    {
        $this->moduleManager->add($module);
        return $this;
    }

    /**
     * Host app hook: configure container before modules boot.
     *
     * Signature:
     *   function(Container $c, SettingsInterface $settings): void
     *
     * @param callable $bootstrap
     * @return $this
     */
    public function withContainerBootstrap($bootstrap)
    {
        $this->containerBootstraps[] = $bootstrap;
        return $this;
    }

    /**
     * Add global middleware to the app.
     *
     * @param MiddlewareInterface $mw
     * @return $this
     */
    public function withMiddleware(\boru\dweb\Contracts\MiddlewareInterface $mw)
    {
        $this->middleware[] = $mw;
        return $this;
    }

    private function boot() {
        $spec = \boru\dweb\Support\KernelSpec::fromConfig($this->config)
        ->withContainer($this->containerBootstrapCallback)
        ->withModules(function ($mm, $c, $cfg) {
            $mm->add(new CoreModule());
            foreach ($this->moduleManager->all() as $m) $mm->add($m);
        })
        ->withMiddleware($this->middleware);

        $this->env = \boru\dweb\Support\KernelBuilder::buildEnv($spec);
    }

    public function getContainer()
    {
        if (!$this->env) $this->boot();
        return $this->env->container();
    }
    public function getContainerOld()
    {
        if (!$this->container) {
            $this->boot();
        }
        return $this->container;
    }

    public function render()
    {
        $req = null;
        try {
            if (!$this->env) $this->boot();

            $req = $this->env->container()->get(\boru\dweb\Http\Request::class);
            $response = call_user_func($this->env->pipeline(), $req);
        } catch (\Exception $e) {
            // Try to log centrally (best effort)
            try {
                if($this->env && $this->env->container()) {
                    $logger = $this->env->container()->get(\boru\dweb\Contracts\LoggerInterface::class);
                    $logger->error('Unhandled exception', array(
                        'exception' => get_class($e),
                        'message' => $e->getMessage(),
                        'file' => $e->getFile(),
                        'line' => $e->getLine(),
                    ));
                }
            } catch (\Exception $ignore) {}

            if ($req === null) {
                try { $req = \boru\dweb\Http\Request::fromGlobals(); } catch (\Exception $ignore) {}
            }
            $debug = (bool)$this->config->get(ConfigKeys::DEBUG_ENABLED, false);
            $response = \boru\dweb\Http\ExceptionResponder::fromException($e, $req, $debug, $this->config);
        }
        try {
            if(!is_object($response)) {
                throw Fail::runtime('Response is not an object',["response"=>$response]);
            }
            \boru\dweb\Http\Emitter::emit($response);
        } catch (\Exception $e) {
            // Last resort: log to error_log
            $response = \boru\dweb\Http\ExceptionResponder::fromException($e, $req, true, $this->config);
            error_log("Failed to emit response: " . $e->getMessage());
            Emitter::emit($response);
        }
    }
}
