<?php

namespace boru\ocr\Confidence;

class ConfidenceReport
{
    /** @var string */
    public $mode = 'tsv'; // tsv | none

    /** @var array */
    public $thresholds = array();

    /** @var array */
    public $overall = array(); // same shape as PageConfidence->toArray(), but "page" omitted

    /** @var PageConfidence[] */
    public $pages = array();

    /** @var Hotspot[] */
    public $hotspots = array();

    /** @var array */
    public $notes = array();

    public function toArray()
    {
        $pages = array();
        foreach ($this->pages as $p) $pages[] = $p->toArray();

        $hotspots = array();
        foreach ($this->hotspots as $h) $hotspots[] = $h->toArray();

        return array(
            'mode' => $this->mode,
            'thresholds' => $this->thresholds,
            'overall' => $this->overall,
            'pages' => $pages,
            'hotspots' => $hotspots,
            'notes' => $this->notes,
        );
    }

    public function toJson($pretty = true)
    {
        return json_encode($this->toArray(), $pretty ? JSON_PRETTY_PRINT : 0);
    }

    public function summaryText($maxHotspots = 10)
    {
        $o = $this->overall;
        $s = array();
        $s[] = "Confidence mode: " . $this->mode;

        if (!empty($o)) {
            $p = isset($o['percentiles']) ? $o['percentiles'] : array();
            $s[] = "Overall: avg=" . $this->fmt($o, 'confAvg')
                . " min=" . $this->fmt($o, 'confMin')
                . " max=" . $this->fmt($o, 'confMax')
                . " p10=" . $this->fmt($p, 'p10')
                . " p50=" . $this->fmt($p, 'p50')
                . " p90=" . $this->fmt($p, 'p90')
                . " lowRatio=" . $this->fmt($o, 'lowConfRatio')
                . " veryLowRatio=" . $this->fmt($o, 'veryLowConfRatio');
        }

        $count = 0;
        foreach ($this->hotspots as $h) {
            if ($count >= $maxHotspots) break;
            $s[] = "Hotspot p{$h->page} {$h->lineKey} avg=" . ($h->confAvg === null ? 'null' : round($h->confAvg, 1))
                 . " lowRatio=" . round($h->lowConfRatio, 2)
                 . " bbox=(".$h->left.",".$h->top.",".$h->width.",".$h->height.")"
                 . " text=\"" . $this->trim($h->sampleText, 120) . "\"";
            $count++;
        }

        foreach ($this->notes as $n) $s[] = "Note: " . $n;

        return implode("\n", $s);
    }

    /**
     * Get a page confidence object by 1-based page number.
     *
     * @param int $page
     * @return PageConfidence|null
     */
    public function page($page)
    {
        $page = (int)$page;
        foreach ($this->pages as $p) {
            if ((int)$p->page === $page) return $p;
        }
        return null;
    }

    /**
     * Return hotspots sorted in their current order (we already sort in calculator),
     * optionally limited to top N.
     *
     * @param int $limit
     * @return Hotspot[]
     */
    public function hotspots($limit = 10)
    {
        $limit = (int)$limit;
        if ($limit <= 0) return array();
        if (count($this->hotspots) <= $limit) return $this->hotspots;
        return array_slice($this->hotspots, 0, $limit);
    }

    /**
     * Hotspots filtered to a specific 1-based page.
     *
     * @param int $page
     * @param int $limit
     * @return Hotspot[]
     */
    public function hotspotsForPage($page, $limit = 10)
    {
        $page = (int)$page;
        $limit = (int)$limit;

        $out = array();
        foreach ($this->hotspots as $h) {
            if ((int)$h->page === $page) {
                $out[] = $h;
                if ($limit > 0 && count($out) >= $limit) break;
            }
        }
        return $out;
    }

    /**
     * A compact associative summary (easy to serialize or inject into prompts).
     *
     * @return array
     */
    public function summaryArray()
    {
        return array(
            'mode' => $this->mode,
            'thresholds' => $this->thresholds,
            'overall' => $this->overall,
            'pageCount' => count($this->pages),
            'hotspotCount' => count($this->hotspots),
        );
    }

    /**
     * Convenience: overall average confidence or null.
     *
     * @return float|null
     */
    public function overallAvg()
    {
        if (!isset($this->overall['confAvg'])) return null;
        return $this->overall['confAvg'];
    }


    private function fmt(array $a, $k)
    {
        if (!array_key_exists($k, $a) || $a[$k] === null) return 'null';
        $v = $a[$k];
        if (is_float($v)) return (string)round($v, 3);
        return (string)$v;
    }

    private function trim($s, $n)
    {
        $s = (string)$s;
        if (strlen($s) <= $n) return $s;
        return substr($s, 0, $n - 3) . '...';
    }
}
