<?php

namespace boru\ocr\Layout\Strategy;

use boru\ocr\Layout\Support\BoundsCalculator;
use boru\ocr\Layout\Support\LayoutDiagnostics;
use boru\ocr\Tesseract\Tsv\TsvRow;

/**
 * Legacy sort: lines sorted by top then left, with optional global multi-column reorder.
 * Includes header protection (do not reorder header band).
 */
class LegacyOrderStrategy implements ReadingOrderStrategyInterface
{
    /** @var BoundsCalculator */
    protected $bounds;

    /** @var bool */
    protected $multiColumn = true;

    /** @var int */
    protected $headerBandPx = 300;

    public function __construct(BoundsCalculator $bounds, array $options = array())
    {
        $this->bounds = $bounds;
        if (array_key_exists('multiColumn', $options)) $this->multiColumn = (bool)$options['multiColumn'];
        if (isset($options['headerBandPx'])) $this->headerBandPx = (int)$options['headerBandPx'];
        if ($this->headerBandPx < 40) $this->headerBandPx = 40;
    }

    public function order(array $lines, LayoutDiagnostics $diag = null)
    {
        foreach ($lines as &$ln) {
            usort($ln, array(__CLASS__, 'cmpLeft'));
        }
        unset($ln);

        usort($lines, array($this, 'cmpLineTopThenLeft'));

        $split = $this->splitHeaderBand($lines);
        $header = $split['header'];
        $rest = $split['rest'];

        if ($this->multiColumn && count($rest) >= 8) {
            $rest = $this->reorderForMultiColumn($rest);
        }

        if ($diag) $diag->strategy = $diag->strategy ? $diag->strategy : 'legacy';

        return array_merge($header, $rest);
    }

    protected function splitHeaderBand(array $lines)
    {
        $minTop = null;
        foreach ($lines as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;
            $t = (int)$b['minTop'];
            if ($minTop === null || $t < $minTop) $minTop = $t;
        }
        if ($minTop === null) return array('header' => array(), 'rest' => $lines);

        $cut = $minTop + (int)$this->headerBandPx;

        $header = array();
        $rest = array();

        foreach ($lines as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;
            if ((int)$b['minTop'] <= $cut) $header[] = $ln;
            else $rest[] = $ln;
        }

        usort($header, array($this, 'cmpLineTopThenLeft'));
        return array('header' => $header, 'rest' => $rest);
    }

    protected function reorderForMultiColumn(array $lines)
    {
        $minX = null;
        $maxX = null;
        $mids = array();

        foreach ($lines as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;
            if ($minX === null || $b['minLeft'] < $minX) $minX = $b['minLeft'];
            if ($maxX === null || $b['maxRight'] > $maxX) $maxX = $b['maxRight'];
            $mids[] = (int)$b['midX'];
        }
        if ($minX === null || $maxX === null || count($mids) < 6) return $lines;

        $width = (int)($maxX - $minX);
        if ($width < 200) return $lines;

        sort($mids);
        $median = $mids[(int)floor(count($mids) / 2)];

        $margin = (int)max(40, $width * 0.12);

        $left = array();
        $right = array();
        $middle = array();

        foreach ($lines as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) { $middle[] = $ln; continue; }
            if ($b['midX'] < $median - $margin) $left[] = $ln;
            elseif ($b['midX'] > $median + $margin) $right[] = $ln;
            else $middle[] = $ln;
        }

        usort($left, array($this, 'cmpLineTopThenLeft'));
        usort($middle, array($this, 'cmpLineTopThenLeft'));
        usort($right, array($this, 'cmpLineTopThenLeft'));

        return array_merge($left, $middle, $right);
    }

    public static function cmpLeft(TsvRow $a, TsvRow $b)
    {
        if ($a->left === $b->left) {
            if ($a->top === $b->top) return 0;
            return ($a->top < $b->top) ? -1 : 1;
        }
        return ($a->left < $b->left) ? -1 : 1;
    }

    public function cmpLineTopThenLeft(array $a, array $b)
    {
        $ba = $this->bounds->lineBounds($a);
        $bb = $this->bounds->lineBounds($b);

        if ($ba === null && $bb === null) return 0;
        if ($ba === null) return 1;
        if ($bb === null) return -1;

        if ($ba['minTop'] === $bb['minTop']) {
            if ($ba['minLeft'] === $bb['minLeft']) return 0;
            return ($ba['minLeft'] < $bb['minLeft']) ? -1 : 1;
        }
        return ($ba['minTop'] < $bb['minTop']) ? -1 : 1;
    }
}
