<?php

namespace boru\ocr\Layout\Strategy;

use boru\ocr\Layout\Support\BoundsCalculator;
use boru\ocr\Layout\Support\LayoutDiagnostics;

/**
 * Mixed layout: header locked + top banded + bottom optionally multi-column (via legacy).
 * Expects a breakY (Y coordinate where bottom region begins).
 */
class MixedLayoutStrategy implements ReadingOrderStrategyInterface
{
    /** @var BoundsCalculator */
    protected $bounds;

    /** @var BandedOrderStrategy */
    protected $banded;

    /** @var LegacyOrderStrategy */
    protected $legacyBottom;

    /** @var int */
    protected $breakY = 0;

    /** @var int */
    protected $headerBandPx = 300;

    public function __construct(BoundsCalculator $bounds, BandedOrderStrategy $banded, LegacyOrderStrategy $legacyBottom, array $options = array())
    {
        $this->bounds = $bounds;
        $this->banded = $banded;
        $this->legacyBottom = $legacyBottom;

        if (isset($options['breakY'])) $this->breakY = (int)$options['breakY'];
        if (isset($options['headerBandPx'])) $this->headerBandPx = (int)$options['headerBandPx'];
        if ($this->headerBandPx < 40) $this->headerBandPx = 40;
    }

    public function withBreakY($breakY)
    {
        $this->breakY = (int)$breakY;
        return $this;
    }

    public function order(array $lines, LayoutDiagnostics $diag = null)
    {
        // Start with banded for stability
        $ordered = $this->banded->order($lines, null);

        $minTop = null;
        foreach ($ordered as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;
            $t = (int)$b['minTop'];
            if ($minTop === null || $t < $minTop) $minTop = $t;
        }
        if ($minTop === null) return $ordered;

        $cut = $minTop + (int)$this->headerBandPx;

        $header = array();
        $rest = array();
        foreach ($ordered as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;
            if ((int)$b['minTop'] <= $cut) $header[] = $ln;
            else $rest[] = $ln;
        }

        $top = array();
        $bottom = array();

        foreach ($rest as $ln) {
            $b = $this->bounds->lineBounds($ln);
            if (!$b) continue;
            if ((int)$b['minTop'] < (int)$this->breakY) $top[] = $ln;
            else $bottom[] = $ln;
        }

        $top = $this->banded->order($top, null);

        // bottom: legacy (may reorder columns)
        $bottom = $this->legacyBottom->order($bottom, null);

        if ($diag) {
            $diag->strategy = 'mixed';
            $diag->metrics['mixedBreakY'] = (int)$this->breakY;
        }

        return array_merge($header, $top, $bottom);
    }
}
