<?php

namespace boru\ocr;

use boru\ocr\Pipeline\OcrResult;
use boru\ocr\Evidence\TsvEvidenceIndex;
use boru\ocr\Format\FrameFormatter;
use boru\ocr\Format\OutputNormalizer;

class OcrEngineResult
{
    /** @var OcrResult */
    protected $result;

    /** @var TsvEvidenceIndex */
    protected $evidence;

    /** @var string|null */
    protected $finalText = null; // reserved for later AI stage

    public function __construct(OcrResult $result, TsvEvidenceIndex $evidence)
    {
        $this->result = $result;
        $this->evidence = $evidence;
    }

    public function confidence()
    {
        return $this->result->confidence;
    }


    /**
     * Access the raw result DTO (all artifacts).
     * @return OcrResult
     */
    public function raw()
    {
        return $this->result;
    }

    /**
     * Access evidence index for callbacks / targeted retrieval.
     * @return TsvEvidenceIndex
     */
    public function evidence()
    {
        return $this->evidence;
    }

    /**
     * Baseline per-page text from TSV layout builder.
     * @return array<int,string>
     */
    public function baselinePages()
    {
        return $this->result->baselinePages;
    }

    /**
     * Baseline framed as a single string (useful for feeding an agent).
     * @return string
     */
    public function baselineFramed()
    {
        return FrameFormatter::framePages($this->result->baselinePages, 'TSV BASELINE OUTPUT');
    }

    /**
     * Plain text OCR pages.
     * @return array<int,string>
     */
    public function textPages()
    {
        return $this->result->textPages;
    }

    /**
     * Plain text framed.
     * @return string
     */
    public function textFramed()
    {
        return FrameFormatter::framePages($this->result->textPages, 'TESSERACT TEXT OUTPUT');
    }

    /**
     * Placeholder for future final AI output.
     * For now, returns baseline framed.
     *
     * @return string
     */
    public function getText()
    {
        $text = $this->finalText ? $this->finalText : $this->baselineFramed();

        return OutputNormalizer::normalize($text, array(
            'collapseSpaces' => true,
            'preserveIndent' => true,
            'collapseBlankLines' => true,
        ));
    }

    /**
     * Later, your AI layer can set this after post-processing.
     *
     * @param string $text
     * @return void
     */
    public function setFinalText($text)
    {
        $this->finalText = (string)$text;
    }

        /**
     * Run AI post-processing using TSV tools and store finalText.
     *
     * @param string $pdfPath
     * @param string $reference
     * @param array $options
     * @return string
     */
    public function runAi($pdfPath, $reference = "", array $options = array())
    {
        $orch = new \boru\ocr\Agent\AiOcrOrchestrator($options);
        $final = $orch->run($pdfPath, $this, $reference);
        $this->setFinalText($final);
        return $final;
    }

}
