<?php

namespace boru\ocr\Page;

class PageImageProviderFactory
{
    /**
     * Create the best available PDF page provider.
     *
     * @param string $pdfPath
     * @param string $imageDir
     * @param array $options Provider options
     * @param callable|null $logger function(string $message):void
     * @return PageImageProviderInterface
     * @throws \Exception
     */
    public static function createBest($pdfPath, $imageDir, array $options = array(), $logger = null)
    {
        $candidates = self::getCandidates();

        foreach ($candidates as $candidate) {
            $class = $candidate['class'];
            $label = $candidate['label'];

            if (!class_exists($class)) {
                continue;
            }

            // If provider exposes isAvailable(), respect it.
            if (method_exists($class, 'isAvailable')) {
                try {
                    if (!call_user_func(array($class, 'isAvailable'))) {
                        continue;
                    }
                } catch (\Exception $e) {
                    // Treat failures as unavailable
                    continue;
                }
            }

            self::log($logger, $label);

            // Constructor signature assumed: ($pdfPath, $imageDir, $options)
            return new $class($pdfPath, $imageDir, $options);
        }

        throw new \Exception("No PDF page image providers available. Install VIPS/MuPDF/Imagick or provide a custom PageImageProvider.");
    }

    /**
     * @return array<int,array{class:string,label:string}>
     */
    protected static function getCandidates()
    {
        return array(
            array(
                'class' => 'boru\\ocr\\Page\\VipsPdfPageImageProvider',
                'label' => 'Using VIPS + MuPDF for PDF rendering & tiling.',
            ),
            array(
                'class' => 'boru\\ocr\\Page\\MuPdfPdfPageImageProvider',
                'label' => 'Using MuPDF for PDF rendering.',
            ),
            array(
                'class' => 'boru\\ocr\\Page\\ImagickPdfPageImageProvider',
                'label' => 'Using Imagick for PDF rendering.',
            ),
        );
    }

    /**
     * @param callable|null $logger
     * @param string $message
     * @return void
     */
    protected static function log($logger, $message)
    {
        if ($logger !== null && is_callable($logger)) {
            call_user_func($logger, $message);
        }
    }
}
