<?php

namespace boru\ocr\Agent;

use boru\ocr\OcrEngineResult;

class AiOcrOrchestrator
{
    /** @var bool */
    protected $usePlanner = true;

    /** @var bool */
    protected $useTableInterpreter = true;

    protected $plannerOptions;
    protected $tableInterpreterOptions;
    protected $finalAgentOptions;

    protected $logger;

    /**
     * @param array $options
     *  - usePlanner (bool) default true
     *  - logger (callable|null) default null
     * @param callable|null $logger
     */
    public function __construct(array $options = array(), callable $logger = null)
    {
        if (array_key_exists('usePlanner', $options)) {
            $this->usePlanner = (bool)$options['usePlanner'];
        }
        if(isset($options['useTableInterpreter'])) {
            $this->useTableInterpreter = (bool)$options['useTableInterpreter'];
        }
        if (isset($options['plannerOptions'])) {
            $this->plannerOptions = $options['plannerOptions'];
        }
        if (isset($options['tableInterpreterOptions'])) {
            $this->tableInterpreterOptions = $options['tableInterpreterOptions'];
        }
        if (isset($options['finalAgentOptions'])) {
            $this->finalAgentOptions = $options['finalAgentOptions'];
        }
        if ($logger !== null) {
            $this->logger = $logger;
        } elseif (array_key_exists('logger', $options)) {
            $this->logger = is_callable($options['logger']) ? $options['logger'] : null;
        } else {
            $this->logger = function ($msg) {
                // Default logger implementation (e.g., log to console or file)
                //echo $msg . PHP_EOL;
            };
        }
    }

    /**
     * @param string $pdfPath
     * @param OcrEngineResult $bundle
     * @param array $plannerOptions
     * @param array $finalAgentOptions
     * @return string final framed output
     */
    public function run($pdfPath, OcrEngineResult $bundle)
    {
        if(!$this->plannerOptions) {
            $this->plannerOptions = new AgentOptions();
            $this->plannerOptions->logger = $this->logger;
        }
        if(!$this->tableInterpreterOptions) {
            $this->tableInterpreterOptions = new AgentOptions();
            $this->tableInterpreterOptions->logger = $this->logger; 
        }
        if(!$this->finalAgentOptions) {
            $this->finalAgentOptions = new AgentOptions();
            $this->finalAgentOptions->logger = $this->logger; 
        }
        // Build TSV tools using the correct FunctionBase/ToolDefinition system.
        $bridge = new EvidenceToolBridge($bundle->evidence());
        $tools = $bridge->buildTools();
        $plan = null;
        if ($this->usePlanner) {
            $planner = new PlannerAgent($pdfPath, $this->plannerOptions, $this->logger);
            $plan = $planner->run($bundle);
        }

        $tableJson = null;
        if ($this->useTableInterpreter) {
            $tableAgent = new TableInterpreterAgent($pdfPath, $tools, $this->tableInterpreterOptions, $this->logger);
            $tableJson = $tableAgent->run($bundle, $plan);
        }

        $final = new FinalAgent($pdfPath, $tools, $this->finalAgentOptions, $this->logger);
        return $final->run($bundle, $plan, $tableJson);
    }
}
